/**************

----------------------------------
Dan Morris
dmorris@cs.stanford.edu
http://techhouse.brown.edu/dmorris
----------------------------------

**************/

#include "stdafx.h"
#include "digitalcar.h"
#include "pt_ioctl.h"
#include <conio.h>
#include <math.h>

CDigitalCar::CDigitalCar() {
  portnum = 0x0;
  bitstate = 0;
  speed = 0.0f;
  turn = 0.0f;
  pulse = 1;
  lastpulse = 0;
  m_initialized = 0;
}


CDigitalCar::~CDigitalCar() {
  uninitialize();
}


// If we're not pulsing the motors to control speed, any
// speed values aboe speed_threshold are rounded up to
// full speed; anything below this value is rounded to 0
#define SPEED_THRESHOLD 0.5

// Any turn value with absolute value greater than this value
// is a full turn, anything less than this is zero turn
#define TURN_THRESHOLD 0.5

// If pulsing is enabled, any speed less than this value 
// turns the motors off
#define SPEED_PULSE_THRESHOLD 0.1


#define PULSE_CYCLE 0.5f

// Handle periodic timer callbacks
void CDigitalCar::timer() {

  // If we're not pulsing or not initialized, don't do anything
  if (pulse == 0 || m_initialized == 0) return;

  // If the current speed is less than our threshold, turn
  // the motors off
  if (fabs(speed) < SPEED_PULSE_THRESHOLD) {
    bitstate &= ~(1<<SPEED_BACK);
    bitstate &= ~(1<<SPEED_FORWARD);
    outportb(portnum,bitstate);
    return;
  }

  // Are we going forward or back?
  int activebit = 
    (speed < 0)?(1<<SPEED_BACK):(1<<SPEED_FORWARD);

  float curtime = ((float)(GetTickCount()))/1000.0f;
  
  // How long has it been since we updated this bit?
  float elapsed = curtime - lastpulse;

  // Is the bit on right now?
  int enabled = ((bitstate & activebit) > 0)?1:0;
  
  // If this pin is on right now, but it's time to turn it off
  if (enabled && (elapsed > fabs(speed) * PULSE_CYCLE) && (fabs(speed) < 1.0) ) {
    lastpulse = curtime;
    bitstate &= ~(1<<SPEED_FORWARD);
    bitstate &= ~(1<<SPEED_BACK);    
    outportb(portnum,bitstate);
  }

  // If the pin is off right now, and it's time to turn it on
  else if ((!enabled) && (elapsed > (1.0f-fabs(speed)) * PULSE_CYCLE)) {
    lastpulse = curtime;
    if (speed > 0) {
      bitstate |= (1<<SPEED_FORWARD);
      bitstate &= ~(1<<SPEED_BACK);
    }
    else {
      bitstate &= ~(1<<SPEED_FORWARD);
      bitstate |= (1<<SPEED_BACK);
    }
    outportb(portnum,bitstate);    
  }
}

float CDigitalCar::set_speed(float speed) {

  // Don't do anything if we're not controlling a car
  if (portnum == 0x0 || m_initialized == 0) return 0.0;

  // If we're allowing speeds that are analog, just update the
  // speed and let the pulsing take care of actually changing
  // the bits
  if (pulse) {
    this->speed = min(speed,1.0f);
    this->speed = max(speed,-1.0f);
    return this->speed;
  }

  // Otherwise turn on or off the right bits
  if (speed > SPEED_THRESHOLD) {
    bitstate &= ~(1<<SPEED_BACK);
    bitstate |= (1<<SPEED_FORWARD);
    this->speed = 1.0;
  }

  else if (
    speed < -SPEED_THRESHOLD) {
    bitstate &= ~(1<<SPEED_FORWARD);
    bitstate |= (1<<SPEED_BACK);
    this->speed = -1.0;
  }

  else {
    bitstate &= ~(1<<SPEED_BACK);
    bitstate &= ~(1<<SPEED_FORWARD);
    this->speed = 0.0;
  }

  if (m_initialized) outportb(portnum,bitstate);

  return this->speed;
}


float CDigitalCar::set_turn(float turn) {

  // If we're turning right...
  if (turn > TURN_THRESHOLD) {
    bitstate &= ~(1<<TURN_LEFT);
    bitstate |= (1<<TURN_RIGHT);
    this->turn = 1.0;
  }

  // If we're turning left...
  else if (turn < -TURN_THRESHOLD) {
    bitstate &= ~(1<<TURN_RIGHT);
    bitstate |= (1<<TURN_LEFT);
    this->turn = -1.0;
  }

  // If we're going straight
  else {
    bitstate &= ~(1<<TURN_RIGHT);
    bitstate &= ~(1<<TURN_LEFT);
    this->turn = 0.0;
  }

  // Update the parallel port
  if (m_initialized) outportb(portnum,bitstate);  

  return this->turn;
}


int CDigitalCar::initialize(int portnum) {

  this->portnum = portnum;

  // Open a connection to the parallel port
  if (OpenPortTalk()==-1) {
    _cprintf("Failed to open porttalk library\n");
    this->portnum = 0;
    return -1;
  }

  m_initialized = 1;

  bitstate = 0x0;

  // Set the parallel port to output mode.
  unsigned char control = inportb(portnum+2);
  control &= (~(1<<5));
  outportb(portnum+2,control);

  // Write 0's to the port
  outportb(portnum,bitstate);

  return 0;
}


int CDigitalCar::uninitialize() {
  m_initialized = 0;
  outportb(portnum,0);
  ClosePortTalk();
  return 0;
}

