/**************

----------------------------------
Dan Morris
dmorris@cs.stanford.edu
http://techhouse.brown.edu/dmorris
----------------------------------

**************/

#include "stdafx.h"

#pragma warning (disable : 4786)
#include "joystickio.h"
#include <math.h>
#include "Dxerr8.h"

/* Called once for each object (axis/button/etc.) found on the first
   system joystick. */
BOOL CALLBACK  EnumJoystickObjectsCallback(const DIDEVICEOBJECTINSTANCE* pdidoi,
                                           VOID* pContext ) {

  // We use the optional data pointer to point to our CJoystickIO
  CJoystickIO* cjio = (CJoystickIO*)(pContext);

  /* For axes that are returned, set the DIPROP_RANGE property for the
     enumerated axis in order to scale min/max values to 1000/-1000. */
  if( pdidoi->dwType & DIDFT_AXIS ) {

   /* _cprintf("Setting axis range for axis %d (type %lx)\n",
      DIDFT_GETINSTANCE(pdidoi->dwType), pdidoi->guidType.Data1); */
  
    DIPROPRANGE diprg; 
    diprg.diph.dwSize       = sizeof(DIPROPRANGE); 
    diprg.diph.dwHeaderSize = sizeof(DIPROPHEADER); 
    diprg.diph.dwHow        = DIPH_BYID; 
    diprg.diph.dwObj        = pdidoi->dwType; // Specify the enumerated axis
    diprg.lMin              = -1000; 
    diprg.lMax              = +1000; 

    // Set the range for the axis
    if(FAILED(cjio->m_pJoystick->SetProperty(DIPROP_RANGE, &diprg.diph))) {
        _cprintf("Error setting range for axis %d\n",
          (int)(DIDFT_GETINSTANCE(pdidoi->dwType)));
    }

  } // if this object is an axis

  // This tells dinput to keep enumerating axes
  return DIENUM_CONTINUE;
}


// Called once for each joystick found
BOOL CALLBACK  EnumJoysticksCallback(const DIDEVICEINSTANCE* pdidInstance,
                                     VOID* pContext ) {

  // We use the optional data pointer to point to our CJoystickIO
  CJoystickIO* cjio = (CJoystickIO*)(pContext);

  cjio->enumerated_devices.push_back(*pdidInstance);

  // Print out the user-friendly joystick name
  
  //_cprintf("Enumerated joystick: %s (%s)\n",
  //  pdidInstance->tszInstanceName,pdidInstance->tszProductName);
  
  return DIENUM_CONTINUE;
}


CJoystickIO::CJoystickIO(HANDLE event_to_signal) {

  this->event_to_signal = event_to_signal;

  active = 0;

  /* Initially map the joysticks's x-axis to x and the joystick's
     y-axis to y. */
  whichAxis[0] = 0;
  whichAxis[1] = 1;
  whichAxis[2] = 2;
  
  // Set up useful scale factors to map [-1000,1000] into [-.5,.5]
  int i;
  for(i=0;i<3;i++) {
    scale_factor[i] = (0.5 / 1000.0);
    offset[i] = 0.0;
  }

  for(i=0;i<JSINDEX_COUNT;i++) {
    pos[i] = 0.0;
  }

  /* Reverse the y-axis by default, since most joysticks
     report pushing forward as negative y. */
  scale_factor[1] *= -1.0;

  control_type = ABSOLUTE_JS_CONTROL;
  
  m_pJoystick = 0;
}


CJoystickIO::~CJoystickIO() {
  releaseDevice();
}


int CJoystickIO::startDevice(char* device_to_use) {

  enumerate_devices();
 
  if (active == 1) {
    
    if (m_pJoystick == 0) {
      _cprintf("Hmmm... no joystick selected?\n");
      return -1;
    }

    DIDEVICEINSTANCE cur;
    cur.dwSize = sizeof(DIDEVICEINSTANCE);
    m_pJoystick->GetDeviceInfo(&cur);

    if (strcmp(cur.tszInstanceName,device_to_use)==0) {
      _cprintf("Device %s (%s) is already active\n",cur.tszInstanceName,
        cur.tszProductName);
      return 0;
    }

    // Okay, we're really going to initialize a new device
    releaseDevice();
  }
  
  active = 0;

  lastx = lasty = prev_lastx = prev_lasty = prev_dt = 0;
  gotInitialState = 0;
  
  // Make sure the axis specifiers set by the GUI are legal
  for(int i=0; i<3; i++) {
    if (whichAxis[i] >= JSINDEX_COUNT) {
      _cprintf("Illegal axis specifier %d for axis %d\n",
        whichAxis[i],i);
      return -1;
    }
  }

  // Record the current device->tg2 axis mappings, for data logging
  m_axis_mask = (whichAxis[0] & 0xff) |
              ((whichAxis[1] & 0xff) << 8);
  
  // If we've already initialized a joystick, we're done
  if (m_pJoystick) {
    _cprintf("CJIO: I think this case should never happen...\n");
    active = 1;
    return 0;
  }

  if (enumerated_devices.size() == 0) {
    _cprintf("Error: could not find any directinput devices...\n");
    return -1;
  }

  HRESULT hr;
  DIDEVICEINSTANCE instance;
  m_pJoystick = 0;

  std::list<DIDEVICEINSTANCE>::iterator iter;
  for(iter = enumerated_devices.begin(); iter != enumerated_devices.end();
    iter++) {
    
    if (device_to_use == 0 || (0==strcmp((*iter).tszInstanceName,device_to_use))) {
      instance = (*iter);
      break;
    }
  }

  if (iter == enumerated_devices.end()) {
    _cprintf("Error: could not find direct input device %s\n",
      device_to_use==0?"[unspecified]":device_to_use);
    return -1;
  }

  _cprintf("Initializing device %s (%s)\n",instance.tszInstanceName,
     instance.tszProductName);
   
  // Obtain an interface to the specified joystick.
  hr = g_pDirectInput->
    CreateDevice(instance.guidInstance, &m_pJoystick, NULL );
  
  // If device creation failed, then we can't use this joystick
  if( FAILED(hr) ) {
    _cprintf("Error initializing device %s (%s)\n",instance.tszInstanceName,
      instance.tszProductName);
      return -1;
  }
  
  // Make sure we got a joystick
  if(m_pJoystick == 0) {
    _cprintf("Error: joystick not found...\n");
    return -1;
  }

  // Set the data format to "simple joystick" - a predefined data format 
  if(FAILED(hr = m_pJoystick->SetDataFormat( &c_dfDIJoystick2 ) ) ) {
    _cprintf("Could not set joystick format...\n");
    releaseDevice();
    return -1;
  }
  
  /* Set the cooperative level to let DInput know how this device should
     interact with the system and with other DInput applications. (In
     this case, we ask for exclusive access...) */
  if(FAILED(hr = m_pJoystick->SetCooperativeLevel(g_mainWnd,
    DISCL_EXCLUSIVE | DISCL_BACKGROUND | DISCL_NOWINKEY))) {
    _cprintf("Error setting joystick cooperating level...\n");
    releaseDevice();
    return -1;
  }

  // Configure the joystick for immediate, absolute mode
  DIPROPDWORD  dipdw; 
  dipdw.diph.dwSize = sizeof(DIPROPDWORD); 
  dipdw.diph.dwHeaderSize = sizeof(DIPROPHEADER); 
  dipdw.diph.dwObj = 0; 
  dipdw.diph.dwHow = DIPH_DEVICE; 

  // Leave the buffer size at 0, for immediate data

#ifdef USE_BUFFERED_IO
  dipdw.dwData = DINPUT_BUFFER_SIZE;
  hr = m_pJoystick->SetProperty(DIPROP_BUFFERSIZE, &dipdw.diph); 
    
  if (hr != DI_OK) {
    if (hr == DI_PROPNOEFFECT) {
      _cprintf("Setting joystick buffer size had no effect...\n");
    }
    else {
      const char* errBuf = DXGetErrorString8(hr);
      _cprintf("Warning: error setting joystick buffer size : %s\n",errBuf);

      /* This isn't really an error; some devices just don't let you
         set the buffer size. */
      /*releaseDevice();
      return -1;*/
    }
  } // if setting the buffer size failed
#endif

  // Now set the joystick to report absolute coordinates
  
  // For reference, if I wanted relative mode, I would say this
  // dipdw.dwData = DIPROPAXISMODE_REL;

  dipdw.dwData = DIPROPAXISMODE_ABS;
  hr = m_pJoystick->SetProperty(DIPROP_AXISMODE, &dipdw.diph); 
  
  // Make sure the device was okay with that...
  if (hr != DI_OK) {
    if (hr == DI_PROPNOEFFECT) {
      _cprintf("Setting joystick axis mode had no effect..\n");
    }
    else {
      _cprintf("Error setting joystick axis mode...\n");
      releaseDevice();
      return -1;
    }
  }
  
  /* Enumerate the joystick objects to set the min/max
     values property for discovered axes. */
  if(FAILED(hr = m_pJoystick->EnumObjects(EnumJoystickObjectsCallback, 
    (VOID*)this, DIDFT_ALL))) {
    _cprintf("Error enumerating joystick objects...\n");
    releaseDevice();
    return -1;
  }

  if (event_to_signal) {
    if (FAILED
       (hr = m_pJoystick->SetEventNotification(event_to_signal))) {
      
      _cprintf("Warning: could not set event notification, error:\n%s\n",
        DXGetErrorDescription8(hr));
      m_polled_device = STATUS_POLL;
    }
    
    else if (hr == DI_POLLEDDEVICE) {
      _cprintf("Using a polled dinput device, enabling 1ms timer\n");
      m_polled_device = STATUS_POLL;
    }

    else {
      _cprintf("Initialized event notificaiton\n");
      m_polled_device = STATUS_NOTIFY;
    }

  }

  /* Now take exclusive control over the joystick and start
     getting data from it. */
  if (FAILED(hr = m_pJoystick->Acquire())) {
    _cprintf("Failed to acquire joystick\n");
    releaseDevice();
    return STATUS_FAILED;
  }
  
  /* Getting the original state of the joystick immediately
     after initializing always results in an error (at least
     on the device I'm using), so I defer this to the first
     pass through Idle(). */
  // if (pollSate() < 0) return -1;
  
  active = 1;

  return m_polled_device;
}


int CJoystickIO::pollState() {

  if (active == 0) return -1;

  HRESULT hr;

  // Flush the device buffer
#ifdef USE_BUFFERED_IO
  DWORD dwItems = INFINITE; 
  hr = m_pJoystick->GetDeviceData( 
    sizeof(DIDEVICEOBJECTDATA), 
    NULL, 
    &dwItems, 
    0); 

  if (FAILED(hr)) {
    _cprintf("Failed to flush dinput buffer...\n");
  }
#endif
  
  // Poll the device to read the current state
  hr = m_pJoystick->Poll(); 
  if(FAILED(hr)) {
     _cprintf("Error polling joystick state...\n");
     if (hr == DIERR_INPUTLOST) {
       _cprintf("Lost input device...\n");
     }
     releaseDevice();
     return -1;
  }

  DIJOYSTATE2 js;

  // Get the input's device state (immediate data mode)
  if(FAILED(hr = m_pJoystick->GetDeviceState(sizeof(DIJOYSTATE2), &js))) {
      _cprintf("Error getting joystick state...\n");
      if (hr == DIERR_INPUTLOST) {
       _cprintf("Lost input device...\n");
     }
     releaseDevice();
      return -1;
  }

  // Copy the current position to pos and lastReportedCoordinates
  long xval = js.lX;
  long yval = js.lY;
  long zval = js.lZ;

  lastReportedCoordinates[0] = pos[0] = xval;
  lastReportedCoordinates[1] = pos[1] = yval;
  lastReportedCoordinates[2] = pos[2] = zval;

  lastReportTime = reportTimer.GetTime();

  // Copy the current button state
  for(int i=0; i<NUM_BUTTONS; i++) {
    // DirectX reports button state in the high bit for some reason
    buttons[i] = (js.rgbButtons[i] & 0x80)?1:0;
  }

  UpdateXY();

  return 0;

} // int CJoystickIO::pollState()

void CJoystickIO::stopDevice() {
  active = 0;
}

void CJoystickIO::releaseDevice() {
  
  DIDEVICEINSTANCE cur;
  if (m_pJoystick) {
    cur.dwSize = sizeof(DIDEVICEINSTANCE);
    m_pJoystick->GetDeviceInfo(&cur);
  }

  _cprintf("Releasing device %s...\n",
    m_pJoystick==0?"[unknown]":cur.tszInstanceName);

  active = 0;

  // Release exclusive control over the joystick
  if (m_pJoystick) m_pJoystick->Unacquire();

  // Release our handle to the dinput device
  SAFE_RELEASE(m_pJoystick);

  m_pJoystick = 0;
}


/* Convert the provided MS joystick-axis-specifier to an
   item in my ordered enumeration, so I can index directly
   into an array with the returned identifier.

   Returns -1 if the id is not found. */
int getJoystickAxisIndex(DWORD dwOfs) {

  switch (dwOfs) {
  case DIJOFS_X :
    return JSINDEX_X;
  case DIJOFS_Y :
    return JSINDEX_Y;
  case DIJOFS_Z :
    return JSINDEX_Z;
  case DIJOFS_RX :
    return JSINDEX_RX;
  case DIJOFS_RY :
    return JSINDEX_RY;
  case DIJOFS_RZ :
    return JSINDEX_RZ;
  case DIJOFS_POV(0):
    return JSINDEX_POV0;
  case DIJOFS_POV(1):
    return JSINDEX_POV1;
  case DIJOFS_POV(2):
    return JSINDEX_POV2;
  case DIJOFS_POV(3):
    return JSINDEX_POV3;
  case DIJOFS_SLIDER(0):
    return JSINDEX_SLIDER0;
  case DIJOFS_SLIDER(1):
    return JSINDEX_SLIDER1;
  default:
    return -1;
  }
}


void CJoystickIO::idle() {

  // Make sure we've already initialized the joystick
  if (!(m_pJoystick)) {
    _cprintf("Error: joystick is null\n");
    return;
  }

  /* On the first pass through idle, we poll the device for its
     initial state. */
  if (!(gotInitialState)) {
    pollState();
    gotInitialState = 1;
  }

  /* If the joystick does not produce any updates for
     VELOCITY_UPDATE_TIME seconds and it's operating in
     velocity mode, the current position will be advanced
     by the current velocity. */
  #define VELOCITY_UPDATE_TIME 0.02

  /* The "velocity control" mode requires that the dot keeps moving
     even when the joystick isn't moving.  So here in Idle() we check 
     whether it's been a while since receiving a js update, in which
     case we move the dot along at constant velocity.  ("The dot" means
     lastx and lasty).*/
  else if (control_type == VELOCITY_JS_CONTROL) {
    double curTime = reportTimer.GetTime();
    if ((curTime - lastReportTime > VELOCITY_UPDATE_TIME) && prev_dt > 0) {

      double old_vx = (lastx - prev_lastx) / prev_dt;
      double old_vy = (lasty - prev_lasty) / prev_dt;

      double dt = curTime - lastReportTime;

      prev_lastx = (float)lastx;
      prev_lasty = (float)lasty;
      prev_dt = (float)dt;

      lastx += (float)(old_vx * dt);
      lasty += (float)(old_vy * dt);

      lastReportTime = curTime;
      
    }
  }

  // Tell dinput to get buffered events from the device
  HRESULT hr = m_pJoystick->Poll(); 
  if(FAILED(hr)) {
     _cprintf("Error polling joystick state...\n");
  }

  // Get the input's device state (buffered data mode)
  DIDEVICEOBJECTDATA rgdod[DINPUT_BUFFER_SIZE]; 
  DWORD dwItems = DINPUT_BUFFER_SIZE; 
  hr = m_pJoystick->GetDeviceData( 
    sizeof(DIDEVICEOBJECTDATA), 
    rgdod, 
    &dwItems, 
    0); 

  if (SUCCEEDED(hr)) { 

    /* Note that a buffer overflow is still considered a 'success'
       by dinput; it just means some data got lost.  Just print
       a warning if this happens. */
    if (hr == DI_BUFFEROVERFLOW) { 
      _cprintf("Warning: joystick buffer overflow\n");
    }

  }

  else {
    _cprintf("Error: failed to read joystick data\n");
    return;
  }

  // For each event we just got from the device
  for(unsigned int i=0;i<dwItems;i++) {

    int val = rgdod[i].dwData; 

    /* Which joystick object (axis/button/etc.) is this 
       an update of? */
    int index = getJoystickAxisIndex(rgdod[i].dwOfs);

    // If this item is in fact a recognized axis
    if (index >= 0) {

      // Store its position
      pos[index] = val;

      // ...but don't do anything else unless we care about this axis
      if (whichAxis[0] != index &&
          whichAxis[1] != index) continue;
    }
    
    // If this item is a button
    else if (rgdod[i].dwOfs >= DIJOFS_BUTTON0 && rgdod[i].dwOfs <= DIJOFS_BUTTON31 &&
      rgdod[i].dwOfs - DIJOFS_BUTTON0 < NUM_BUTTONS) {
      buttons[rgdod[i].dwOfs-DIJOFS_BUTTON0] = (rgdod[i].dwData & 0x80)?1:0;
    }

    // If this is not a button or a recognized axis
    else continue;

    /* Update the representation of device position that's in
       scaled TG2 coordinates. */
    UpdateXY();


  } // for each event we got from the device

} // void CJoystickIO::idle()


void CJoystickIO::UpdateXY() {

  if (control_type == ABSOLUTE_JS_CONTROL) {
    lastx = (float)(pos[(whichAxis[0])] * scale_factor[0] + offset[0]);
    lasty = (float)(pos[(whichAxis[1])] * scale_factor[1] + offset[1]);
  }

  else if (control_type == VELOCITY_JS_CONTROL) {

    double curTime = reportTimer.GetTime();

    /* Compute the time that elapsed since we last computed
       TG2 coordiantes. */
    double dt = (curTime - lastReportTime);
    lastReportTime = curTime;

    if (fabs(pos[(whichAxis[0])]) < MOVEMENT_THRESHOLD) pos[(whichAxis[0])] = 0;
    if (fabs(pos[(whichAxis[1])]) < MOVEMENT_THRESHOLD) pos[(whichAxis[1])] = 0;

    prev_lastx = lastx;
    prev_lasty = lasty;
    prev_dt = (float)dt;

    // change_in_tg2_coordinates = joystick_position * velocity * time
    lastx += (float)(pos[(whichAxis[0])] * scale_factor[0] * dt);
    lasty += (float)(pos[(whichAxis[1])] * scale_factor[1] * dt);
    
  }

  else if (control_type == RELATIVE_JS_CONTROL) {

    /* Add the change since the last time we cmoputed TG2
       coordinates. */
    float dx = (float)
      ((pos[(whichAxis[0])] * scale_factor[0]) -
       (lastReportedCoordinates[(whichAxis[0])] * scale_factor[0]));
    lastx += dx;

    float dy = (float)
      ((pos[(whichAxis[1])] * scale_factor[1]) -
       (lastReportedCoordinates[(whichAxis[1])] * scale_factor[1]));
    lasty += dy;

    /* Store the coordinates we just used to compute the
       new position. */
    for(int i=0;i<3;i++) lastReportedCoordinates[i] = pos[i];
  }
} // void CJoystickIO::UpdateXY()


void CJoystickIO::GetXY(float* x,float* y) {

  if (x) *x = lastx;
  if (y) *y = lasty;

}


int CJoystickIO::enumerate_devices() {

  enumerated_devices.clear();

  HRESULT hr;
  
  // Initialize directinput if necessary
  if (g_pDirectInput == 0) {
    if( FAILED( hr = DirectInput8Create( GetModuleHandle(NULL), DIRECTINPUT_VERSION, 
      IID_IDirectInput8, (VOID**)&g_pDirectInput, NULL ) ) ) {
      _cprintf("Failed to create direct input interface...\n");
      return -1;
    }
  }

  /* Enumerate joysticks; this call will set the m_pjoystick
     property of this class to point to the newly created
     joystick device. */
  if (FAILED( hr = g_pDirectInput->EnumDevices(
                                   DI8DEVCLASS_GAMECTRL,
                                   // DI8DEVCLASS_ALL,
                                   EnumJoysticksCallback,
                                   (void*)this, DIEDFL_ATTACHEDONLY))) {
    _cprintf("Error enumerating joysticks...\n");
    return -1;
  }

  return 0;
}