/**************

----------------------------------
Dan Morris
dmorris@cs.stanford.edu
http://techhouse.brown.edu/dmorris
----------------------------------

**************/


/***********

  Defines a nice wrapper around the DirectInput interface to
  game devices.  Provides access to axis positions and buttons.

***********/


#ifndef _JOYSTICKIO_H_
#define _JOYSTICKIO_H_

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <windows.h>
#include "globals.h"
#include "celapsed.h"

#include <list>
using std::list;

// startdevice() (described below) returns one of :
typedef enum joy_status {
  STATUS_FAILED=-1, STATUS_NOTIFY, STATUS_POLL
};
  
/* Assume anything less than 50/1000 isn't "real" movement;
   joysticks tend to be very noisy around 0 and tend not
   to center well. */
#define MOVEMENT_THRESHOLD 50
    
/* Nice wrappers around delete and release(), to make sure
   we don't operate on null pointers. */
#define SAFE_DELETE(p)  { if(p) { delete (p);     (p)=NULL; } }
#define SAFE_RELEASE(p) { if(p) { (p)->Release(); (p)=NULL; } }

// Need to define the d-input version before including dinput.h
#ifndef DIRECTINPUT_VERSION
#define DIRECTINPUT_VERSION 0x0800
#endif

#include <dinput.h>

// The number of game device packets that we'll ask dinput to buffer
#define DINPUT_BUFFER_SIZE 100

// The maximum number of buttons that we'll query the device for
#define NUM_BUTTONS 10

/* Uncomment to use buffered I/O instead of immediate I/O.  Only
   absolute mode is supported for immediate I/O right now. */
// #define USE_BUFFERED_IO

/* We can map from the absolute positions returned by the joystick
   to TG2 coordinates in several ways... */
enum joystick_position_types {

  /* Assumes that wherever the device was when startDevice()
     was called is position 0, and computes positions relative
     to that.  Ignores the 'offset' variable. */
  RELATIVE_JS_CONTROL = 0,

  /* One joystick position maps to exactly one TG2 position.  Uses
     the 'scale' and 'offset' variables as described below. */
  ABSOLUTE_JS_CONTROL,

  /* Holding the joystick in the "up" position makes the TG2
     position move up, etc. */
  VELOCITY_JS_CONTROL
};

/* An ordering of the types of axes that can appear on a
   directinput joystick....  MS provides definitions for each
   of these, but they don't guarantee any order, so you can't
   just use them as indices.  So I convert MS's constants to this
   ordered set of axis labels. */
typedef enum {
    JSINDEX_X=0, JSINDEX_Y, JSINDEX_Z,
    JSINDEX_RX, JSINDEX_RY, JSINDEX_RZ,
    JSINDEX_SLIDER0, JSINDEX_SLIDER1,
    JSINDEX_POV0, JSINDEX_POV1, JSINDEX_POV2, JSINDEX_POV3,
    JSINDEX_COUNT
} joystick_axis_indices;

static const char* joystick_axis_names[] = {
    "Joystick X", "Joystick Y", "Joystick Z",
    "Rotational X", "Rotational Y", "Rotational Z",
    "Slider 1", "Slider 2", "Slider 3",
    "POV hat 1", "POV hat 2", "POV hat 3", "POV hat 4"
};

class CJoystickIO {
  
public:

  // non-zero if this device has been initialized by some target
  int active;

  /* On our first pass through Idle(), we poll the device for an
     initial position.  This flag tells us that we've already
     completed that first polling operation. */
  int gotInitialState;

  HANDLE event_to_signal;

  /* Each position in this array represents one screen space
  axis, and the value stored there represents the device axis
  mapped to that TG2 axis .  So if the x-axis on the joystick
  controlled all screen space axes, this would look like [0,0,0]. */
  int whichAxis[3];
  
  // the scale factor corresponding to each screen-space axis
  double scale_factor[3];
  double offset[3];

  /*
    For example, in absolute mode, the x value that is reported to
    TG2 is computed as :

     *x = pos[(whichAxis[0])] * scale_factor[0] + offset[0];

     ...where 'pos' is the array of raw x,y,z values obtained
     from the device.

     'offset' is ignored in relative mode.
  */

  /* The mode in which device coordinates are mapped to TG2
     coordinates.  Uses the joystick_position_types enumeration. */
  int control_type;

  /* The joystick-space coordinates that were current as of the last
     time GetXY was called. */
  double lastReportedCoordinates[3];

  /* Used for computing the net change between
     calls to GetXY when in relative mode. */
  CElapsed reportTimer;
  double lastReportTime;
  float prev_lastx,prev_lasty,prev_dt;
  
  /* The current position of the joystick, in joystick coordinates,
     in the joystick's native axes.  Units will be scaled from -1000
     to +1000 (we'll tell dinput to use that system).
  
     We support this many axes to allow for device components that are
     represented as sliders, rudders, rotational devices, etc.  Axes
     are ordered in the same order in which they appear in the 
     joystick state structure (DIJOYSTATE) :
  
     x,y,z
     rotational x,y,z
     slider 1,2
     point-of-view 1,2,3,4
     */
  double pos[JSINDEX_COUNT];

  // The current state of each of the joystick's buttons
  unsigned char buttons[NUM_BUTTONS];

  /* Should be called at the beginning of each experiment to
     start up the device.  Note that the device is not closed
     after each experiment; we keep an open handle to dinput as
     long as TG2 is running. */

  virtual int startDevice(char* device_to_use = 0);

  // should be called at the end of each experiment
  virtual void stopDevice();

  /* should be called by the main loop as often as is possible
     to check the queue for joystick device events. 
  
     Use only if you're using buffered I/O... see the constant at the
     top of this file.  If you're not using buffered I/O, use pollState()
     instead. */
  virtual void idle();

  // releases the handle to dinput
  virtual void releaseDevice();

  /* Polls the device for information.  We use this once at the
     beginning of each experiment, to capture the initial state of
     the joystick.  In general, we'll used buffered device I/O instead
     of polled device I/O. */
  virtual int pollState();

  // Our handle to the dinput device
  LPDIRECTINPUTDEVICE8 m_pJoystick;

  CJoystickIO(HANDLE event_to_signal = 0);
  virtual ~CJoystickIO();

  // 1 if this device must be polled, 0 if it is doing event notifications
  int m_polled_device;

  /* The most recent coordinates obtained from the device, 
     in the TG2 coordinate system. */
  float lastx;
  float lasty;

  /* Concisely represents the joystick axes being used to
     control TG2 axes.  The joystick axis used to control x
     is in bits 0-7, y bits 8-15, z bits 16-23. 
  
     Initialized when startDevice() is called. */
  int m_axis_mask;

  /* Called internally to update lastx and lasty when 
     joystick events are receivd. */
  virtual void UpdateXY();

  /* The usual GetXY method, called by targets to access lastx
     and lasty.  Also called internally to get the most recent
     TG2 coordinates for data logging. */
  virtual void GetXY(float* x,float* y);

  // all joysticks enumerated at the last enumeration run
  list<DIDEVICEINSTANCE> enumerated_devices;

  // Repopulates the enumerated_devices list
  virtual int enumerate_devices();
};

#endif
