%
% function [ textlines, letterchoices ] = gentext(alphabet,text)
%
% Given an alphabet array (which contains spline representations of all the
% available letters, in the form generated by the 'makealphabet' script), this 
% script generates line segments that represent the letters in the given body of text.
%
% The second return value, letterchoices, is a sequence of indices representing which
% example of each letter was chosen.  This is useful if you generate text and see that
% one letter is really bad, and you want to know which one it is so you can remove it from
% your alphabet.  So if 'text' is the string 'abc', this return value will have three elements,
% one indicating which 'a' I chose, one indicating which 'b' I chose, and one indicating
% which 'c' chose.
%
% A warning is generated if the script encounters any letters or characters in 'text'
% that are not present in 'alphabet'.
%
% The output is a series of line segments, suitable for feeding to the 'drawlines' script
% for display on the screen and/or printing.
%
% This script handles the following aspects of handwriting synthesis :
%
% * randomly choosing letter splines from the available samples for each letter
%
% * randomly varying the points in each chosen sample, so even letters generated from the
%   same sample won't look the same.
%
% * randomly varying the spacing between letters
%
% * breaking lines at appropriate places
%
% * randomly varying the baseline of the text
%
% Pagination is not handled yet.
%
% Depending on the scale of the letters you defined in your alphabet, you may need to change
% some of the magic numbers I put at the top of this script.  They are appropriate for letters
% generated from the sample bitmap enclosed with these scripts, so if you scan at that resolution,
% or scale to that resolution, they should work for you.
%
function [ textlines, letterchoices ] = gentext(alphabet,text)
letterchoices = [];

% This defines the width of the 'page' and the horizontal width at which a line break will
% be inserted.
MAXWIDTH = 3500;
BREAKWIDTH = 3300;

% The maximum amplitude of the cosine wave that will be used to vary the baseline
% of the letters on each line (randomly chosen for each line, with this amplitude as the max).
MAXAMPLITUDE = 25;

% The maximum deviation that is actually allowed from the baseline (the cosine wave generated
% using some amplitude <= MAXAMPLITUDE is clipped to this value).
MAXBASELINE = 15;

% When it's time to break lines, we increment the current y position by a random
% amount between LINEINCA and (LINEINCA + LINEINCB)
LINEINCA = 140;
LINEINCB = 40;

% When we encounter a 'space' character, increment the x position by a random
% amount between SPACEINCA and (SPACEINCA + SPACEINCB)
SPACEINCA = 40;
SPACEINCB = 25;

initialtime = cputime;
splinetime = 0;
lettertime = 0;

if (nargin < 2)
  fprintf(1,'function [ textlines ] = gentext( alphabet,text )');
  return;
end

% textpoints will be an array of lines, of the form :
% [x1 y1 x2 y2]

% Keep track of the current x and y positions on the page
curx = 20;
liney = 1;


xs = 1:1:MAXWIDTH;

% Generate a randomly varying baseline
amplitude = rand(1) * MAXAMPLITUDE;
frequency = (1/(MAXWIDTH*2)) + (1/(MAXWIDTH/2)) * rand(1) * 2;
phase = rand(1) * 2 * pi;

yfunc = amplitude * sin(frequency * xs + phase);
yfunc = yfunc - yfunc(1);

for(i=1:length(yfunc))
  if (yfunc(i) < -1*MAXBASELINE) yfunc(i) = -1*MAXBASELINE; end;
  if (yfunc(i) > MAXBASELINE) yfunc(i) = MAXBASELINE; end;
end

% plot(xs,yfunc);

textlines = [];
linecount = 0;

for(i=1:length(text))
  
  % Choose the y coordinate for the current letter
  if (curx < MAXWIDTH)
      cury = liney + ceil(yfunc(ceil(curx)));
  else
      cury = liney + ceil(yfunc(ceil(MAXWIDTH)));
  end
  
  % The current letter
  cur = text(i) + 0;
  
  c = sprintf('\n');
  
  % Is it time for a line break?  If so, increment the y position,
  % generate a new baseline, and reset the x position.
  if (cur == c | ((cur == ' ' + 0) & (curx > BREAKWIDTH)))
     liney = cury - (LINEINCA + rand(1) * LINEINCB);
     
     amplitude = rand(1) * MAXAMPLITUDE;
     frequency = (1/(MAXWIDTH*2)) + (1/(MAXWIDTH/2)) * rand(1) * 2;
     phase = rand(1) * 2 * pi;

    yfunc = amplitude * sin(frequency * xs + phase);
    yfunc = yfunc - yfunc(1);

    for(i=1:length(yfunc))
     if (yfunc(i) < -1*MAXBASELINE) yfunc(i) = -1*MAXBASELINE; end;
     if (yfunc(i) > MAXBASELINE) yfunc(i) = MAXBASELINE; end;
    end
    
    curx = 20;
    
    continue;
  end
  
  % Move the x position over if we encounter an ASCII space character
  if (cur == ' ' + 0)
    curx = curx + SPACEINCA + rand(1) * SPACEINCB;
    continue;
  end
  
  if (cur > length(alphabet) | isempty(alphabet{cur}))
    fprintf(1,'Warning : alphabet does not contain %c\n',text(i));
    continue;
  end
  
  len = length(alphabet{cur});
  
  % Randomly choose a spline for the current letter, from the available set
  % of sample splines for this letter.
  
  % Is there more than one sample available for this letter?
  if (len > 1)
    r = -1;
    while(r < 0)
      r = ceil(rand(1)*len);
    end
    curletter = alphabet{cur}(r);
    letterchoices(end+1) = r;
  else
    curletter = alphabet{cur};
    letterchoices(end+1) = 1;
  end
  
  % Now we have a letter struct, we need to make a spline from it
  
  curtime = cputime;
  s = splineletter(curletter);
  splinetime = splinetime + cputime - curtime;
  
  % s is now a cell array of double arrays, containing spline points
  
  minx = 10000;
  maxx = 0;
  % first find the minimum x in the letter
  for(j=1:length(s))
    curpoints = s{j};
    lminx = min(curpoints(1,:));
    lmaxx = max(curpoints(1,:));
    
    minx = min(lminx,minx);
    maxx = max(lmaxx,maxx);
  end
  
  letterwidth = maxx - minx;
  
  % now actually make the letter
  
  curtime = cputime;
  
  % for each spline that defines this letter, make line segments from
  % successive points
  for(j=1:length(s))
    curpoints = s{j};
    
    len = length(curpoints);
    x1vals = curpoints(1,1:len-1)' + curx - minx;
    x2vals = curpoints(1,2:len)' + curx - minx;
    y1vals = curpoints(2,1:len-1)' + cury;
    y2vals = curpoints(2,2:len)' + cury;
    
    newlines = [x1vals y1vals x2vals y2vals];
    
    linecount = linecount + length(curpoints);
    
    textlines = [textlines; newlines];
    
  end
  lettertime = lettertime + cputime - curtime;
  
  curx = curx + letterwidth + ceil(rand(1)*10) + 6;
end

totaltime = cputime - initialtime;

fprintf(1,'Total time: %f\nFraction in splines:%f\n',totaltime,splinetime/totaltime);
fprintf(1,'Fraction making letters %f\n',lettertime/totaltime);