% function [ lettertable ] = getletters( filename )
%
% Takes an image file and opens it up in a figure window.  Lets you quickly define 
% bounding rectangles for letters, and automatically pages around the image
% to make the process easier.  You just have to click the upper-left and lower-right
% corners of bounding boxes around any letter that you like.
% 
% After each letter that you enter, it will ask you what letter you just bounded.
% Type that into the matlab shell, or hit 'q' to quit or 'return' to move the viewing
% window to another area on the page.
%
% Try to get a few good samples from each letter.  Don't worry if you mess up; you
% can remove 
%
% The output is a list of bounding boxes for each example, to be passed to the 'getpoints'
% script, where you'll actually define the contours for each letter example.
%
function [ lettertable ] = getletters( filename )

if (nargin < 1)
    fprintf(1,'Filename is required...\n');
    return;
end

imgdata = imread(filename);

if (length(size(imgdata)) < 3)
    imgdata = repmat(imgdata,[1 1 3]);
end

if (~(exist('imgdata')))
    fprintf(1,'Error reading image file %s\n\n',filename);
    return;
end

lettercount = 1;

rows = size(imgdata,1);
cols = size(imgdata,2);

BLOCKSIZE = 700;

initrow = 300;
initcol = 1;

currow = initrow;
curmaxrow = BLOCKSIZE + currow;

curcol = initcol;
curmaxcol = BLOCKSIZE + currow;

OFFSET = 100;

while(currow < rows - OFFSET)

    if (currow < 1)
        currow = 1;
    end
    
    if (curmaxrow > rows)
        curmaxrow = rows;
        currow = rows - BLOCKSIZE;
    end
    
    curcol = 1;
    curmaxcol = BLOCKSIZE;
    
    while(curcol < cols - OFFSET)
        
        if (curcol < 1)
            curcol = 1;
        end
        
        if (curmaxcol > cols)
            curmaxcol = cols;
            curcol = cols - BLOCKSIZE;
        end
        
        fprintf(1,'Displaying (%d,%d) to (%d,%d)\n',currow,curcol,curmaxrow,curmaxcol);
        
        curdata = imgdata(currow:curmaxrow,curcol:curmaxcol,:);
        
        image(curdata);
        colormap(gray);
        
        hold on;
        
        quit = 0;
        
        while(quit == 0)
          [x,y] = ginput(2);
          line([x(1) x(1)],[y(1) y(2)]);
          line([x(1) x(2)],[y(2) y(2)]);
          line([x(2) x(2)],[y(1) y(2)]);
          line([x(2) x(1)],[y(1) y(1)]);        
          
          in = input('What letter was that (return for next frame, quit to quit)? ','s');
          
          if (strcmp(in,'quit') == 1) return; end;
          if (length(in) == 0) break; end;
          
          lettertable{lettercount} = struct('letter',in,'points',...
            [curcol+x(1),currow+y(1),curcol+x(2),currow+y(2)]);
          
          lettercount = lettercount + 1;
              
        end
        
        curcol = curmaxcol - OFFSET;
        curmaxcol = curcol + BLOCKSIZE;
        
        hold off;
    end
    
    currow = curmaxrow-50;
    curmaxrow = currow + BLOCKSIZE;
end
        
hold off;
close all;
