/* 

   KBDDRIVEWIN
   Dan Morris
   dmorris@cs.stanford.edu

   Application to monitor the serial port interface to the Stowaway
   Palm keyboard and generate keyboard events.
   
   Just plug the serial cable in (connected as is described
   below) and run this application.  The mapping from palm keys to keyboard
   events is contained in 'palmkey.ini', which is included in this distrubtion.
   There is also a separate application that you can use to build your own mapping.

   The cable is connected as follows... if the keyboard is oriented as if you
   were typing on it, let pin 1 be on the far left.

   Keyboard pin   |  Serial Port Pin
   ---------------------------------
   2            |  DTR
   3            |  RD
   4            |  RTS
   7            |  GND
   10           |  GND
   --------------------------------

   This program uses two threads, one to watch the serial port and one
   to sit in the windows event loop, basically waiting for clicks on the
   taskbar icon that tell the program to sleep or quit.

   Double-click the taskbar icon to kill the application, right-click it to
   suspend the application.

*/
   

#include <Winuser.h>
#include "stdafx.h"
#include "conio.h"
#include "stdio.h"
#include "stdlib.h"
#include "Windows.h"
#include "Winable.h"
#include "shellapi.h"
#include "resource.h"

// The hard-coded scan code for the 'function' key on the keyboard
#define FUNCKEY 0x22
#define ICONMSG 15001

char szAppName[] = "palm_kbd";

/* Information about an individual key, read in from
   the configuration file. */
typedef struct keyinfo {
	int vkey;
	int scankey;
	char name[20];
} keyinfo;

int palm, vkey, scan;
char name[50];
HWND     hWnd;
HINSTANCE ghinst;
DCB dcb;

HICON b_h_icon;
HANDLE hComm;
unsigned char buffer[1024];
int bytesread;
char c;
keyinfo keys[256];

// Which serial port are we using
int g_port;

/* Callback for the dialog menu, which allows you to choose a
   serial port. */
BOOL CALLBACK dlgproc(HWND hwnd, UINT Message, WPARAM wParam, LPARAM lParam) {

	HWND item;
	int xsize, ysize;

	switch(Message)
   {
		
	  
	  /* Create the dialog window. */
      case WM_INITDIALOG:
		  g_port = 1;
		  item = GetDlgItem(hwnd, IDC_COM1);
		  SendMessage(item, BM_CLICK, 0, 0);
		  xsize = GetSystemMetrics(SM_CXSCREEN) - 100;
		  ysize = GetSystemMetrics(SM_CYSCREEN) - 100;

		  SetWindowPos(hwnd, HWND_TOP, xsize / 2, ysize / 2, 0, 0, SWP_NOSIZE);

		  return TRUE;
      case WM_COMMAND:
		  
		  switch(HIWORD(wParam)) {
		
		  // Someone clicked a button!
		  case BN_CLICKED :
			  switch(LOWORD(wParam)) {
			  case IDB_OK:
				  EndDialog(hwnd,g_port);
				  return TRUE;
			  case IDB_CANCEL:
				  EndDialog(hwnd, -1);
				  return TRUE;
			  default:
				  g_port = LOWORD(wParam) - IDC_COM1 + 1;
				  return TRUE;
			  } // end "which button"
		  } // end "which command
		  default :
			  return FALSE;
	} // end "which message"

   return FALSE;
} // end dlgproc



FILE* input;
int active;
HANDLE activateEvent;

/* Call to drop RTS temporarily and reset the keyboard, since
   it goes to sleep once in a while. */
void resetkbd() {
	dcb.fRtsControl = RTS_CONTROL_DISABLE;
	SetCommState(hComm,&dcb);
	Sleep(500);
	dcb.fRtsControl = RTS_CONTROL_ENABLE;
	SetCommState(hComm,&dcb);
}

// Turn on or off the taskbar icon
void activateIcon(int addicon) {

	b_h_icon = LoadIcon(ghinst,MAKEINTRESOURCE(
		active?KB_ICON:KB_X_ICON));
	NOTIFYICONDATA nid;
	nid.cbSize = sizeof(NOTIFYICONDATA);
	nid.hWnd = hWnd;
	nid.uID = 15000;
	nid.uFlags = NIF_ICON | NIF_MESSAGE | NIF_TIP;
	nid.uCallbackMessage = ICONMSG;
	nid.hIcon = b_h_icon;
	strcpy(nid.szTip,"Palm kbd control");
	int action;
	if (addicon == 0) action = NIM_MODIFY;
	else if (addicon == 1) action = NIM_ADD;
	else if (addicon == 2) action = NIM_DELETE;
	Shell_NotifyIcon(action,&nid);
}


LRESULT CALLBACK b_WndProc(HWND hwnd, UINT Message, WPARAM wParam, LPARAM lParam)
{
   switch(Message)
   {
      case WM_CLOSE:
         DestroyWindow(hwnd);
      break;
      case WM_DESTROY:
		 activateIcon(2);
         PostQuitMessage(0);
	  break;
	  case ICONMSG:
		  if (lParam == WM_LBUTTONDBLCLK) {
				DestroyWindow(hwnd);
		  }
		  else if (lParam == WM_RBUTTONDOWN || lParam == WM_RBUTTONDBLCLK) {
			  active = (1-active);
			  activateIcon(0);
			  if (active) {
				  resetkbd();	  
				  SetEvent(activateEvent);
			  }
		  }
	  break;
      default:
         return DefWindowProc(hwnd, Message, wParam, lParam);
   }
   return 0;
}

/* A thread that constantly monitors the serial port for
   keyboard events. */
DWORD WINAPI readThread(LPVOID data) {

	char comname[10];

	// Open the serial port
	hComm = CreateFile("COM5",GENERIC_READ | GENERIC_WRITE, 0, 0, OPEN_EXISTING,
		FILE_ATTRIBUTE_NORMAL, 0);

	while (hComm == INVALID_HANDLE_VALUE) {

		// Ask the user which serial port to use
		int i = DialogBox(ghinst, MAKEINTRESOURCE(IDD_DIALOG1),
			hWnd, dlgproc);
		if (i == -1) SendMessage(hWnd,WM_DESTROY,0,0);
		sprintf(comname,"COM%d",i);

		hComm = CreateFile(comname,GENERIC_READ | GENERIC_WRITE, 0,
			0, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
	}

	FillMemory(&dcb,sizeof(dcb),0);
	
	dcb.DCBlength = sizeof(dcb);
	dcb.BaudRate = 9600;
	dcb.fParity = TRUE;
	dcb.fOutxCtsFlow = FALSE;
	dcb.fOutxDsrFlow = FALSE;
	dcb.fDtrControl = DTR_CONTROL_ENABLE;
	dcb.fDsrSensitivity = FALSE;
	dcb.fTXContinueOnXoff = TRUE;
	dcb.fOutX = FALSE;
	dcb.fInX = FALSE;
	dcb.fErrorChar = FALSE;
	dcb.fNull = FALSE;
	dcb.fRtsControl = RTS_CONTROL_DISABLE;
	dcb.fAbortOnError = FALSE;
	dcb.wReserved = 0;
	dcb.XonLim = 0;
	dcb.XoffLim = 0;
	dcb.Parity = NOPARITY;
	dcb.StopBits = ONESTOPBIT;
	dcb.XonChar = 0xa;
	dcb.XoffChar = 0xb;
	dcb.ErrorChar = 0;
	dcb.EofChar = EOF;
	dcb.EvtChar = '\0';
	dcb.ByteSize = 8;

	// Make sure the keyboard isn't asleep
	resetkbd();

	COMMTIMEOUTS cto = {0};

	cto.ReadIntervalTimeout = MAXDWORD;
	cto.ReadTotalTimeoutMultiplier = MAXDWORD; //0; //MAXDWORD;
	// If you sit idle for ten minutes, reset the kbd
	cto.ReadTotalTimeoutConstant = 6000; // 0; // 1000;
	cto.WriteTotalTimeoutConstant = 0;
	cto.WriteTotalTimeoutMultiplier = 0;
	
	SetCommTimeouts(hComm, &cto);

	SetupComm(hComm, 1024, 1024);

	// What is the state of the function key
	int func_on = 0;

	while(1) {

		// Wait for a keyboard event
		if (!active) WaitForSingleObject(activateEvent,INFINITE);
		
		ReadFile(hComm,buffer,1024,(unsigned long *)(&bytesread),0);

		if (!active) continue;

		/* Right now I'm only interested in the first character... */
		if (bytesread > 0) {
			unsigned char c = buffer[0];
			
			// Mark that the function key was toggled
			if ( (c&(~(1<<7))) == FUNCKEY) {
				if (c&(1<<7)) func_on = 0;
				else func_on = 1;
				continue;
			}

			WORD wVk, wScan;

			// Get the input code
			int incode = c&(~(1<<7));

			/* If this should be a numerical function key, map it to the
			   scan code range for function keys. */
			if ( func_on && (incode<2)) {
				incode+=0xf1;
			}

			else if ( func_on && (incode < 8) && (incode > 3) ) {
				incode+=(0xf1 - 1);
			}

			else if ( func_on && (incode>33) && (incode<37) ) {
				incode+= (0xf8 - 34);
			}

			// Now look up this input code in our big table of codes...
			wVk = keys[incode].vkey;
			if (wVk == 0) {
				continue;
			}

			// ...and convert it to the Windows keyboard code.
			wScan = keys[incode].scankey;

			// Now put together a keypress event and send it to Windows
			tagINPUT ti;
			ti.type = INPUT_KEYBOARD;
			ti.ki.wVk = wVk;
			ti.ki.wScan = wScan;
			ti.ki.dwFlags = (c&(1<<7))?KEYEVENTF_KEYUP:0;
			ti.ki.time = 0;
			ti.ki.dwExtraInfo = 0;

			int val = SendInput(1,&ti,sizeof(tagINPUT));
			
		}

		// if we got 0 bytes, reset the keyboard
		else {
			resetkbd();
		}

	}

}


int APIENTRY WinMain(HINSTANCE hInstance,
                     HINSTANCE hPrevInstance,
                     LPSTR     lpCmdLine,
                     int       nCmdShow)
{
	//AllocConsole();
	active = 1;

	activateEvent = CreateEvent(NULL,FALSE,FALSE,NULL);

	/* First read in the information from palmkey.ini, the 
	   file created by the 'calibration' utility. */
	ghinst = hInstance;
	input = fopen("palmkey.ini","r");	
	if(input == NULL) {
		MessageBox(hWnd,"Couldn't open input file!","Input file error",
			MB_ICONEXCLAMATION | MB_OK | MB_SETFOREGROUND);
		exit(1);
	}
	FillMemory(&keys,sizeof(keys),0);

	if (fscanf(input,"Palm\tvkey\tscan\tname\n") == EOF) {
		MessageBox(hWnd,"Invalid input file!","Input file error",
			MB_ICONEXCLAMATION | MB_OK | MB_SETFOREGROUND);
		exit(1);

	}

	while(fscanf(input,"%x\t%d\t%d\t%s\n",&palm,&vkey,&scan,name) > 0) {
		keys[palm].scankey = scan;
		keys[palm].vkey = vkey;
		strcpy(keys[palm].name,name);
	}
	
	// Now I've read all the data, I'll launch the serial port reader	

	DWORD tid;
	CreateThread(0, 0, readThread, 0, 0, &tid);

	MSG msg;
	WNDCLASS wc;

	if (hPrevInstance == NULL) {
        // Setup and register a window class for our main window.
        wc.hCursor          = LoadCursor(NULL, IDC_ARROW);
        wc.hIcon            = NULL;
        wc.lpszMenuName     = NULL;
        wc.lpszClassName    = szAppName;
        wc.hbrBackground    = (HBRUSH)(GetStockObject(BLACK_BRUSH));
        wc.hInstance        = hInstance;
        wc.style            = CS_HREDRAW | CS_VREDRAW;
        wc.lpfnWndProc      = b_WndProc;
        wc.cbClsExtra       = 0;
        wc.cbWndExtra       = 0;
    
        if(! RegisterClass(&wc))
            return FALSE;   
    }

    // Create a full-screen window with no title bar or scroll bars.
    hWnd = CreateWindow(szAppName, szAppName, WS_POPUP, 0, 0,
        GetSystemMetrics(SM_CXSCREEN), GetSystemMetrics(SM_CYSCREEN),
        NULL, NULL, hInstance, NULL);

	// Put the taskbar icon on the taskbar
	activateIcon(1);

    while(GetMessage(&msg, NULL, 0, 0))
    {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }

	return 0;
}

