/*
   LIBLCD
   Dan Morris
   dmorris@cs.stanford.edu

   This is a small DLL that will make it easy for other
   applications to write data out to an optrex-type LCD
   module. 

   This program links against the 'portio' library, which I'm
   including with this distribution.  This library allows
   us to access the parallel port in win2k.  You don't need to link
   against it if you're only compiling in win98.

   Before using liblcd :

   * Open up the portio zipfile and put the libraries in your library
     path.
   * If you're in win2k, install the portio driver

   To use this liblcd :

   * first call setparport(address) with the address of your parallel 
     port.  Not strictly necessary in win98 if your parallel port is at
	 0x378.
   * call init_lcd()
   * start using outstring(char*) and outchar(char) to write interesting
     things to your lcd.

   A .pdf file is included with this distribution; this file contains
   information about the interface to optrex-type LCD's.

   The mapping between parallel port pins and optrex lcd pins is as 
   follows :

   PIN MAPPINGS :
   PARPORT		LCD		FUNCTION	PARPORT BIT
   1			6		Clock		Control 0 (inverted)
   2-9			7-14	Data 0-7	Data 0-7
   10
   11
   12
   13
   14			5		Read/write	Control 1 (inverted)
   15
   16			4		Reg select	Control 2
   17
   18-25		1		Ground

   LCD pins 2 and 3 (vcc and vee) should be externally connected to
   5V to power the LCD.

*/

*/

#include "stdafx.h"
#include "liblcd.h"
#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <windows.h>

#define DEVICE_NUMBER 3

// Use LPT1 by default
int g_curport = 0x378;
int g_curdevice = 0;

/* Use #defines to decide whether we're in win98 or win2k.  In
   win98, we can use outb to write to the paralell port, in win2k
   we go through the 'portio' driver. */
#ifdef WIN2K
#include "APICDECL.H"
#endif

/* Write wrappers around the input and output functions that will take
   appropriate actions depending on the operating system.  This way I can
   just call port_in or port_out from the code, without worrying about
   which low-level API it's going to use. */
int port_in(short whichRegister) {
#ifdef WIN2K
	int status;
	int toreturn = Peek8(g_curdevice, whichRegister, &status);
	return toreturn;
#else
	return _inp(g_curport+whichRegister);
#endif
}

int port_out(short whichRegister, int data) {
#ifdef WIN2K
	int status;
	int toreturn = Poke8(g_curdevice, whichRegister, data, &status);
	return toreturn;
#else
	return _outp(g_curport+whichRegister, data);
#endif
}

int g_numchars;

BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved
					 )
{
	switch (ul_reason_for_call)
	{
		case DLL_PROCESS_ATTACH:
		case DLL_THREAD_ATTACH:
		case DLL_THREAD_DETACH:
		case DLL_PROCESS_DETACH:
			break;
    }
    return TRUE;
}


// Actually clocks out our data (or reads and returns it)
int clock_out(int data, bool READ, bool RS) {

	// Start with the clock off (pin is inverted)
	int control = 1;

	// Set up the read/write pin on the parallel port, ON to read
	if (READ) {
		_cprintf("Reading from the port.\n");
		control |= (1<<5);
	}

	// Set up the read/write pin on the LCD, ON to read (but pin is inverted)
	if (READ) control &= ~(1<<1);

	// Set up the RS pin on the LCD (not inverted)
	if (RS) control |= (1<<2);

	if (!(READ)) {
		port_out(0, data);
		//_outp(PARPORT, data);
	}

	port_out(2,control);
	//_outp(PARPORT+2,control);
	Sleep(5);
	
	// clock that shit
	control &= (~(1));
	port_out(2, control);
	//_outp(PARPORT+2, control);
	Sleep(5);

	// de-clock that shit
	control |= 1;
	port_out(2, control);
	//_outp(PARPORT+2, control);
	Sleep(5);

	// If necessary, read the data
	if (READ) {
		//return _inp(PARPORT);
		return port_in(0);
	}

	else return 0;
}

// The whole process of writing an ASCII character out to the LCD
LIBLCD_API void outchar(char c) {
	if (g_numchars == 20) {
		clock_out(0xc0,0,0);
	}

	else if (g_numchars == 40) {
		clock_out(0x94,0,0);
	}

	else if (g_numchars == 60) {
		clock_out(0xd4,0,0);
	}

	else if (g_numchars == 80) {
		g_numchars = 0;
		clock_out(0x80,0,0);
	}

	g_numchars++;

	
	clock_out(c,0,1);
}

// Write a character string out to the LCD
LIBLCD_API void outstring(char* str) {
	int len = strlen(str);

	for(int i=0; i<len; i++) {
		outchar(str[i]);
	}
}

// Move the cursor back to the 'home' position on the LCD
LIBLCD_API void home_lcd() {
	clock_out(0x01, 0, 0);
	clock_out(0x02, 0, 0);
	clock_out(0x80, 0, 0);
	g_numchars = 0;
}

// _must_ be called before initializing the lcd
LIBLCD_API void setparport(int addr) {
	g_curport = addr;
#ifdef WIN2k
	g_curdevice = DEVICE_NUMBER;
	mapDeviceToPort(g_curdevice, addr, 3);
#endif
}

LIBLCD_API void init_lcd() {
	g_numchars = 0;
	
	port_out(2, 0);
	//_outp(PARPORT+2, 0);
	Sleep(100);

	// Init function sets
	clock_out(0x30, 0, 0);
	clock_out(0x30, 0, 0);
	clock_out(0x30, 0, 0);

	// Function set
	clock_out(0x3c , 0, 0);

	// Display OFF
	clock_out(0x08, 0, 0);

	// Clear display
	clock_out(0x01, 0, 0);

	// Return home
	clock_out(0x02, 0, 0);

	// Entry mode set
	clock_out(0x06, 0, 0);
	//clock_out(0x05, 0, 0);

	// Display ON
	clock_out(0x0c, 0, 0);
}


