/******
 *
 * Written by Dan Morris
 * dmorris@cs.stanford.edu
 * http://techhouse.brown.edu/~dmorris
 *
 * You can do anything you want with this file as long as this header
 * stays on it and I am credited when it's appropriate.
 *
 ******/

/************
 * 
 * This library allows routing sound files to specific audio devices and/or multiple audio
 * devices.  It's meant to be simple, so you'll have to experiment to find which device is 
 * which on your machine, but device 0 should always be the default.  The most interesting 
 * function is:
 *
 * int build_graph_for_file(const char* filename, int device_mask);
 *
 * Let's say I want to play the file "test.mp3" on devices 0 and 2.  I would say:
 *
 * int handle = build_graph_for_file("test.mp3", 0x5);
 *
 * The last parameter is a bitmask; in binary 0x5 looks like 00000101, i.e. bits
 * "0" and "2" are turned on.  I get back a handle I can pass to all the other
 * functions to start and stop playback of this file on the specified devices; I
 * do most of that with the media_control() function.
 *
 * Also, you should use the init_dshow() and cleanup_dshow() functions to fire up
 * and shut down the library.
 *
 ************/

#ifndef _MULTIDEV_MP3_H_
#define _MULTIDEV_MP3_H_

namespace multidev_mp3 {

/***
  One-time directshow initialization.  No harm done in calling
  it more than once.  Returns 0 for success, -1 for error.
***/
int init_dshow();
int dshow_initialized();

/***
 Cleans up directshow pointers for all active graphs.
***/
int cleanup_dshow();


/***
  Checks all active graphs for notification events.
***/
int handle_all_graph_events();


/***
  Prepares the specified file for playback on the specified devices.

  Each bit in 'device_mask' represents one device.  If bit 0 is
  high, device 0 is included in the graph, etc.

  Returns -1 if there's an error, or a handle to the completed graph
  otherwise.
***/
int build_graph_for_file(const char* filename, int device_mask);


/*** 
  Performs a particular operation on the specified graph; operations
  are defined in the media_control_operations enumeration;

  Returns 0 for success, ERROR_GRAPH_NOT_FOUND for graph not found,
  -1 for other failure.

  Stopping a graph removes its resources, pausing it just temporarily
  freezes it in place.  Parameter is a value in seconds unless otherwise
  noted.
***/
int media_control(int graph_handle, int operation, double parameter=0.0);


/***
  Gets the duration of the audio file referred to by graph_handle, in
  seconds.  Returns -2.0 if the handle is invalid, -1.0 for any other error.
***/
double get_duration(int graph_handle);

/***
  Gets the current position of the audio file referred to by graph_handle, in
  seconds.  Returns -2.0 if the handle is invalid, -1.0 for any other error.
***/
double get_current_position(int graph_handle);

}; // end of the multidev_mp3 namespace

typedef enum {

  MEDIA_CONTROL_PLAY=0,
  MEDIA_CONTROL_STOP,
  MEDIA_CONTROL_PAUSE,

  // Positive numbers are relative to file start, negative
  // numbers are relative to file end.
  MEDIA_CONTROL_SEEK,
  MEDIA_CONTROL_SET_STOP_POSITION,

  // Positive numbers are forward, negative numbers are
  // backward.
  MEDIA_CONTROL_SKIP,

  // No operation, just a status check
  MEDIA_CONTROL_NULL,

  // Adjust the volume for this graph (all outputs)...
  //
  // The parameter should range from 0 (silence) to 1.0 (full volume)
  //
  // All graphs are created at full volume (1.0).
  MEDIA_CONTROL_SET_VOLUME,

  // Adjust the pan for this graph (all outputs)...
  //
  // The parameter should range from -1.0 (left) to 1.0 (right).
  //
  // All graphs are created at neutral pan (0.0).
  MEDIA_CONTROL_SET_PAN
    
} media_control_operations;


#define ERROR_GRAPH_NOT_FOUND -2

#endif