/*****
 * 
 * multi_snd_jni.java
 *
 * Dan Morris
 * dmorris@cs.stanford.edu
 * http://techhouse.brown.edu/dmorris
 *
 * You can do anything you want with this file as long as this 
 * header stays on top and I get credit where it's appropriate.
 *
 *****/

/*****
 *
 * This class provides an interface for playing wave files, playing
 * mp3 or other media files, or synthesizing text on multiple audio
 * 
 * It uses native methods defined in the multi_snd_jni.dll library, so
 * you'll need to have that in your working directory or your path.
 * 
 * There are several functions you care about :
 *
 * int NumDevices() will tell you how many sound devices are available.
 *
 * int PlaySound(filename, device_mask) actually plays a .wav file.
 *
 * int SpeakText(text, device_mask, purge) synthesizes some text.
 *
 * int BuildGraphForFile(String filename, int device_mask);
 *
 * int MediaControl(int graph_handle, int operation, double parameter);
 *
 * * The latter two functions are used to play media files via directshow,
 *   and are documented in the second header comment below.
 *
 * The main(...) function provides an example of how to use each
 * of these functions.
 *
 * PlaySound() takes a filename and a device mask.  The filename
 * specifies what .wav file you want to play, and the device mask
 * specifies which devices you want to play it on.  Each bit in 
 * the device mask corresponds to a device; if a bit is high, we'll
 * play the file on that device.
 *
 * For example, to play "bob.wav" on just device zero, do :
 *
 * MultiSoundPlayer msp = new MultiSoundPlayer();
 * msp.PlaySound("bob.wav", (1<<0) );
 *
 * To play the same file on just device one, do :
 *
 * msp.PlaySound("bob.wav", (1<<1) );
 *
 * To play on both devices, do :
 *
 * msp.PlaySound("bob.wav", (1<<0) | (1<<1) );
 *
 * PlaySound() returns 0 if all goes well, -1 if there's an error.
 * 
 * The device_mask argument to the SpeakText(...) function works the same
 * way.  The additional argument to this function - "purge" - specified
 * whether the speech interface should purge pending speech requests (purge =
 * 1) and speak this text immediately (otherwise it gets queued behind 
 * previous requests).
 *
 * Notes:
 *
 * If you request playback on a device that doesn't exist, nothing 
 * bad happens and no error is returned.
 * 
 * All sounds in progress will stop playing when your application
 * exits.
 * 
 * PlaySound() returns 0 if all goes well, -1 for an error.
 *
 * You can pass '-1' for the device mask to request that a
 * sound be played on all available devices.
 *
 *****/


/***
  Prepares the specified file for playback on the specified devices.

  Each bit in 'device_mask' represents one device.  If bit 0 is
  high, device 0 is included in the graph, etc.

  Returns -1 if there's an error, or a handle to the completed graph
  otherwise.
***/
// int BuildGraphForFile(String filename, int device_mask);


/*** 
  Performs a particular operation on the specified graph; operations
  are defined in the media_control_operations enumeration;

  Returns 0 for success, -1 for operation failure,
  or ERROR_GRAPH_NOT_FOUND if the graph can't be found (for example
  if it was deleted because playback completed).

  Stopping a graph removes its resources, pausing it just temporarily
  freezes it in place.  Parameter is a value in seconds unless otherwise
  noted.
***/
// int MediaControl(int graph_handle, int operation, double parameter);

/***
  Gets the duration of the audio file referred to by graph_handle, in
  seconds.  Returns -2.0 if the handle is invalid, -1.0 for any other error.
***/
// double GetDuration(int graph_handle);

/***
  Gets the current position of the audio file referred to by graph_handle, in
  seconds.  Returns -2.0 if the handle is invalid, -1.0 for any other error.
***/
// double GetCurrentPosition(int graph_handle);

/*****

typedef enum {

  MEDIA_CONTROL_PLAY=0,
  MEDIA_CONTROL_STOP,
  MEDIA_CONTROL_PAUSE,

  // Positive numbers are relative to file start, negative
  // numbers are relative to file end.
  MEDIA_CONTROL_SEEK,
  MEDIA_CONTROL_SET_STOP_POSITION,

  // Positive numbers are forward, negative numbers are
  // backward.
  MEDIA_CONTROL_SKIP,

  // No operation, just a status check
  MEDIA_CONTROL_NULL,

  // Adjust the volume for this graph (all outputs)...
  //
  // The parameter should range from 0 (silence) to 1.0 (full volume)
  //
  // All graphs are created at full volume (1.0).
  MEDIA_CONTROL_SET_VOLUME,

  // Adjust the pan for this graph (all outputs)...
  //
  // The parameter should range from -1.0 (left) to 1.0 (right).
  //
  // All graphs are created at neutral pan (0.0).
  MEDIA_CONTROL_SET_PAN

} media_control_operations;

*****/


public class MultiSoundPlayer {
 
  public static final int MEDIA_CONTROL_PLAY = 0;
  public static final int MEDIA_CONTROL_STOP = 1;
  public static final int MEDIA_CONTROL_PAUSE = 2;
  public static final int MEDIA_CONTROL_SEEK = 3;
  public static final int MEDIA_CONTROL_SET_STOP_POSITION = 4;
  public static final int MEDIA_CONTROL_SKIP = 5;
  public static final int MEDIA_CONTROL_NULL = 6;
  public static final int MEDIA_CONTROL_SET_VOLUME = 7;
  public static final int MEDIA_CONTROL_SET_PAN = 8;

  public static final int ERROR_GRAPH_NOT_FOUND = -2;

  // I declare these private just in case later I want
  // to pass more state data between java and C++.  Users
  // should use public functions, declared below.
  private native int play_sound_multi(String filename, int device_mask);

  private native int
    speak_text_multi(String text, int device_mask, int purge);

  private native int build_graph_for_file(String filename, int device_mask);

  private native int media_control(int graph_handle, int operation,
    double parameter);

  private native int get_num_devices();

  private native double get_duration(int graph_handle); 

  private native double get_current_position(int graph_handle); 

  // We need to load the .dll before we can do anything...
  static {

    System.loadLibrary("multi_snd_jni");

  }

  public MultiSoundPlayer() { }

  public synchronized int SpeakText(String text, int device_mask, int purge) {

    if (device_mask == -1) device_mask = 0xffff;

    return speak_text_multi(text, device_mask, purge);

  }

  public synchronized int PlaySound(String filename, int device_mask) {

    if (device_mask == -1) device_mask = 0xffff;

    return play_sound_multi(filename,device_mask);

  }

  public synchronized int BuildGraphForFile(String filename, int device_mask) {
  
    if (device_mask == -1) device_mask = 0xffff;

    return build_graph_for_file(filename,device_mask);

  }

  public synchronized int MediaControl(int graph_handle, int operation, double parameter) {

    return media_control(graph_handle,operation,parameter);

  }

  public synchronized double GetCurrentPosition(int graph_handle) {
  
   return get_current_position(graph_handle);

  }

  public synchronized double GetDuration(int graph_handle) {
 
   return get_duration(graph_handle);

  }

  public synchronized int NumDevices() {

    return get_num_devices();

  }

  public static void main(String[] args) throws Exception {

    // Create a sound player
    MultiSoundPlayer msp = new MultiSoundPlayer();

    // How many devices are there?
    int num_devices = msp.NumDevices();

    System.out.println("Found " + 
      Integer.toString(num_devices) + " wave devices.");

    String[] mp3_filenames = {
     "mp3s\\15.dmorris.bannf.mp3",
     "mp3s\\17 - Her Majesty.mp3"
    };

    // This just forces initialization of the dshow interface... 
    // initialization would happen whenever you first accessed
    // the library, but doing it here puts some annoying printouts 
    // up front, which is nice.
    System.out.println("Initializing library");
    msp.MediaControl(-1,MEDIA_CONTROL_PLAY,0.0);
    System.out.println("");

    for(int cur_device = 0; cur_device < num_devices; cur_device++) {

      System.out.println("Playing to device " + Integer.toString(cur_device));

      // To play wav files via winmm
      // msp.PlaySound("demo.wav", (1<<cur_device) );

      // To synthesize text...
      // msp.SpeakText("This is a secret", (1<<cur_device), 1);

      int handle = msp.BuildGraphForFile(mp3_filenames[0],(1<<cur_device));

      if (handle < 0) {
       System.out.println("Failed to build mp3 graph");
      }
     
      // Tell the mp3 to stop when it gets to time 4s
      msp.MediaControl(handle,MEDIA_CONTROL_SET_STOP_POSITION,4.0);

      // Play the mp3
      msp.MediaControl(handle,MEDIA_CONTROL_PLAY,0.0);

      // Wait a bit...
      Thread.sleep(1000);

      // Set volume...
      msp.MediaControl(handle,MEDIA_CONTROL_SET_VOLUME,0.85);

      // Wait a bit...
      Thread.sleep(1000);

      // Set pan...
      msp.MediaControl(handle,MEDIA_CONTROL_SET_PAN,-1.0);

      // Tell the mp3 to skip to a position one second before the end
      // msp.MediaControl(handle,MEDIA_CONTROL_SEEK,-1.0);

      // Get the duration of this mp3
      double duration = msp.GetDuration(handle);

      System.out.println("Duration is " + Double.toString(duration));

      // This is how you can test whether a graph has finished
      // playing.  Note that of course while(true) is not a good idea
      // in any useful program, but you can check for completion
      // on all active graphs whenever you go through your main 
      // loop.
      while(true) {
        int result = msp.MediaControl(handle,MEDIA_CONTROL_NULL,0.0);
        if (result == ERROR_GRAPH_NOT_FOUND) break;
        Thread.sleep(1000);
        double curpos = msp.GetCurrentPosition(handle);

        System.out.println("Current position is " +
          Double.toString(curpos));
      
      }

      // Thread.sleep(3000);

    }

    System.out.println("Press a key...\n");
    System.in.read();
    
  }

}

