/*****
 * 
 * multi_snd_core.cpp
 *
 * Dan Morris
 * dmorris@cs.stanford.edu
 * http://techhouse.brown.edu/dmorris
 *
 * You can do anything you want with this file as long as this 
 * header stays on top and I get credit where it's appropriate.
 *
 *****/

/*****
 *
 * This file defines one key function :
 *
 * int play_wave_sound(char* filename, int device_mask)
 *
 * 'filename' specifies a wave file to play
 *
 * device_mask specifies which devices it should be played on.
 * If bit 0 is high, the specified sound is played on device 0,
 * etc.
 *
 * Sounds are loaded entirely into memory before playback starts,
 * and buffers are shared across playback devices.  Playback is 
 * asynchronous, and sound data buffers are freed when playback is
 * finished on all devices.
 *
 * The function returns -1 if there's an error, 0 if all goes well.
 *
 *****/
#include "stdafx.h"

#include "multi_snd_core.h"
#include <windows.h>
#include <mmsystem.h>
#include <stdio.h>
#include <conio.h>

// Some constants used to find the actual data in wave files
#define OFFSET_FORMATTAG        20
#define OFFSET_CHANNELS         22
#define OFFSET_SAMPLESPERSEC    24
#define OFFSET_AVGBYTESPERSEC   28
#define OFFSET_BLOCKALIGN       32
#define OFFSET_BITSPERSAMPLE    34
#define OFFSET_WAVEDATA         44
#define HEADER_SIZE             OFFSET_WAVEDATA


// A structure used to track wave file buffers that might be shared among
// multiple devices.
struct wave_info {

  char* buf;
  int n_playing;
  CRITICAL_SECTION lock;

};


// This gets called every time a wave file finishes playing
void CALLBACK wave_callback(
  HWAVEOUT hwo,      
  UINT uMsg,         
  DWORD dwInstance,  
  DWORD dwParam1,    
  DWORD dwParam2     
  ) {

  if (uMsg == WOM_OPEN) {
    
  }

  // Okay, a wave file finished playing
  else if (uMsg == WOM_DONE) {
    
    // This is the wave header that just finished
    WAVEHDR* phdr = (WAVEHDR*)dwParam1;
    
    // When we create every wave header, we keep a pointer to the 'info' structure
    // associated with it
    wave_info* info = (wave_info*)(phdr->dwUser);

    // Are we _finished_ with this wave data?
    int deleting = 0;

    // Take a lock here... we want exactly one thread to perform the deletion,
    // and it's not clear to me that this callback couldn't be called from
    // multiple threads.
    EnterCriticalSection(&(info->lock));

    info->n_playing--;

    // If no one else is using this, we're done...
    if (info->n_playing == 0) {
      deleting = 1;
    }

    LeaveCriticalSection(&(info->lock));

    // Now do the deletion if necessary...
    if (deleting) {

      // Delete wave data
      delete [] info->buf;

      // Delete the lock
      DeleteCriticalSection(&(info->lock));

      // Delete the meta-data
      delete info;      

    }

    // Each sound has a "private" device handle and wave header, so we can
    // delete those.
    //
    // Note that for some reason, it has to be done in this order, or funny stuff
    // happens.
    waveOutUnprepareHeader(hwo,phdr,sizeof(WAVEHDR));
    delete phdr;
    waveOutClose(hwo);    
    
  }

  else if (uMsg == WOM_CLOSE) {
    
  }

}

// Play the specified wave file on the devices specified in 'device_mask' (each
// bit in 'device_mask' corresponds to one device).  If a requested device is not
// available, nothing bad happens and no error is returned.
//
// Returns -1 for an error, 0 if all goes well
int play_wave_sound(const char* filename, int device_mask) {

  // Open the file and read .wav data
  HANDLE file = CreateFile(filename,GENERIC_READ,FILE_SHARE_READ,0,OPEN_EXISTING,0,0);

  if (file == INVALID_HANDLE_VALUE) {
    printf("Could not open file %s\n",filename);
    return -1;
  }

  DWORD size = GetFileSize(file,0);

  if (size == 0) {
    printf("Could not get file size...\n");
    return -1;
  }

  // Allocate a giant buffer for .wave data
  char* buf = new char[size];

  // Read data into the buffer
  DWORD bytes_read;
  int result = ReadFile(file, buf, size, &bytes_read, NULL);

  if (result == 0) {
    printf("Could not read file %s...\n",filename);
    delete [] buf;
    return -1;
  }

  CloseHandle(file);

  // Get format information from the file data
  WAVEFORMATEX wfex; 
  
  wfex.wFormatTag      = *((WORD* )(buf + OFFSET_FORMATTAG     ));
  wfex.nChannels       = *((WORD* )(buf + OFFSET_CHANNELS      ));
  wfex.nSamplesPerSec  = *((DWORD*)(buf + OFFSET_SAMPLESPERSEC ));
  wfex.nAvgBytesPerSec = *((DWORD*)(buf + OFFSET_AVGBYTESPERSEC));
  wfex.nBlockAlign     = *((WORD* )(buf + OFFSET_BLOCKALIGN    ));
  wfex.wBitsPerSample  = *((WORD* )(buf + OFFSET_BITSPERSAMPLE )); 
  
  // Create our metadata to keep track of the data buffer
  wave_info* info = new wave_info;
  info->buf = buf;
  InitializeCriticalSection(&(info->lock));  
  info->n_playing = 0;
  
  unsigned int num_devices = waveOutGetNumDevs();

  MMRESULT mmresult;

  // Loop over every device and see whether we want to play the sound on that
  // device...
  for(unsigned int cur_device = 0; cur_device < num_devices; cur_device++) {
  
    // If we don't want to play this sound on this device, keep going...
    if ((device_mask & (1 << cur_device)) == 0) continue;

    // Create and initialize a wave header to give to this device
    WAVEHDR* whdr = new WAVEHDR;  

    whdr->lpData = buf + OFFSET_WAVEDATA;
    whdr->dwBufferLength = size - HEADER_SIZE;
    whdr->dwFlags = 0;
    whdr->dwUser = (DWORD)(info);

    // Keep track of the number of devices using this data buffer
    info->n_playing++;
    
    // Open the sound device
    HWAVEOUT hWaveOut;

    mmresult = waveOutOpen(&hWaveOut,cur_device,&wfex,(DWORD)wave_callback,
      0, // callback parameter
      CALLBACK_FUNCTION);

    if (mmresult != MMSYSERR_NOERROR) {
      printf("Could not open device %d\n",cur_device);
      continue;
    }

    // Prepare the sound buffer
    mmresult = waveOutPrepareHeader(hWaveOut, whdr, sizeof(WAVEHDR));

    if (mmresult != MMSYSERR_NOERROR) {

      printf("Could not prepare header for device %d\n",cur_device);

      if (mmresult == MMSYSERR_INVALHANDLE) printf("Invalid handle\n");
      if (mmresult == MMSYSERR_NODRIVER) printf("Driver error\n");
      if (mmresult == MMSYSERR_NOMEM) printf("Memory error\n");

      continue;

    }

    // Send it off to the media system.
    //
    // Note that I don't do any locking here to ensure that a callback doesn't
    // get called to delete the data _while_ I'm initializing other buffers.
    // Very unlikely, but a potential point of problems.
    mmresult = waveOutWrite(hWaveOut, whdr, sizeof(WAVEHDR));

    if (mmresult != MMSYSERR_NOERROR) {

      printf("Could not write wave to device %d\n",cur_device);

      if (mmresult == MMSYSERR_INVALHANDLE) printf("Invalid handle\n");
      if (mmresult == MMSYSERR_NODRIVER) printf("Driver error\n");
      if (mmresult == MMSYSERR_NOMEM) printf("Memory error\n");
      if (mmresult == WAVERR_UNPREPARED) printf("Unprepared\n");

      continue;
    }

  }

  return 0;
}

