/*****
 * 
 * multi_speech.cpp
 *
 * Dan Morris
 * dmorris@cs.stanford.edu
 * http://techhouse.brown.edu/dmorris
 *
 * You can do anything you want with this file as long as this 
 * header stays on top and I get credit where it's appropriate.
 *
 *****/

/*****
 *
 * This file defines the CMultiSpeaker class, used to connect to the
 * MS speech SDK and generate speech on specific interfaces.
 *
 *****/

#include "stdafx.h"

#ifdef INCLUDE_MULTI_SPEECH

#include "multi_speech.h"
#include <conio.h>
#include <sphelper.h>
#include <stdio.h> 

int CMultiSpeaker::initialize() {

  initialized = 0;
  
  // We need to initialize COM before we can use the speech API.
  if(FAILED(::CoInitialize(0))) {
		_cprintf("Speaker: unable to initialize COM\n");
		return -1;
	}

  HRESULT hr;

  // First we need to enumerate available output devices...

  // Get a category that can enumerate devices
  ISpObjectTokenCategory *cpSpCategory;

	hr = SpGetCategoryFromId(SPCAT_AUDIOOUT, &cpSpCategory);

  if (FAILED(hr)) {
    _cprintf("Could not get audio out category\n");
    return -1;
  }

  // Enumerate the actual devices
	IEnumSpObjectTokens *cpSpEnumTokens;

	hr = cpSpCategory->EnumTokens(NULL, NULL, &cpSpEnumTokens);

  if (FAILED(hr)) {
    _cprintf("Could not enumerate speech devices\n");
    return -1;
  }

  hr = cpSpEnumTokens->GetCount(&num_devices);

  if (FAILED(hr)) {
    _cprintf("Could not get number of speech devices\n");
    return -1;
  }
  
  _cprintf("Found %d speech output devices.\n",num_devices);

  if (num_devices == 0) {
    _cprintf("No speech devices found\n");
    return -1;
  }

  int first_device_to_enumerate = 0;

  // Allocate an array of speech objects
  pVoices = new ISpVoice*[num_devices];
  memset(pVoices,0,num_devices*sizeof(ISpVoice*));

  
  // For each available device...
  for(unsigned int dev=first_device_to_enumerate; dev<num_devices; dev++) {

    ISpObjectToken *pSpTok;
  
    hr = cpSpEnumTokens->Item(dev,&pSpTok);

    if (FAILED(hr)) {
      _cprintf("Could not get speech device %d\n",dev);
      uninitialize();
      return -1;
    }
    
    // A handle to the voice COM object
    ISpVoice* cur_voice;
  
    // Create a voice object for this sound device
    hr = CoCreateInstance(CLSID_SpVoice, NULL, CLSCTX_ALL, IID_ISpVoice, (void **)&cur_voice);

    if (FAILED(hr)) {
      _cprintf("Unable to create voice object %d\n",dev);
      uninitialize();
      return -1;
    }   

    // Set this guy's speech to this output device
		hr = cur_voice->SetOutput(pSpTok,FALSE);
    if (FAILED(hr)) {
      _cprintf("Speech error: unable to set sound device to dev...\n",dev);
      uninitialize();
      return -1;
    }
    
    // It seems that this is necessary once to get speech working on
    // some systems.  It's also a good test to make sure everything
    // worked okay.
    
    // But actually, this seems to hang java, so I took it out...
    /*
    hr = cur_voice->Speak(L"", SPF_ASYNC, NULL);

    if (FAILED(hr)) {
      _cprintf("Speech error: unable to speak null sound on dev...\n",dev);
      uninitialize();
      return -1; 
    }
    */
     
		pSpTok->Release();

    pVoices[dev] = cur_voice;
			    
	}

  cpSpEnumTokens->Release();
	cpSpCategory->Release();

  initialized = 1;
  return 0;
    
}


int CMultiSpeaker::uninitialize() {

  // Delete any voice objects we allocated
  if (pVoices) {
    for(unsigned int i=0; i<num_devices; i++) {
      if (pVoices[i]) {
        pVoices[i]->Release();
      }
    }
    delete [] pVoices;
  }

  ::CoUninitialize();

  initialized = 0;

  return 0;
}


CMultiSpeaker::CMultiSpeaker() {

  initialized = 0;
  num_devices = 0;
  pVoices = 0;

  initialize();
}


CMultiSpeaker::~CMultiSpeaker() {

  initialized = 0;

}


int CMultiSpeaker::speak_w(int device_mask, const unsigned short* text_w, int purge) {

  if (initialized == 0) return -1;

  // For each available device...
  for(unsigned int cur_device = 0; cur_device < num_devices; cur_device++) {
  
    // Make sure the caller wants to send speech to this device
    if ( (device_mask & (1<<cur_device)) == 0) continue;

    ISpVoice* pVoice = pVoices[cur_device];

    // Do the actual speaking, purging if requested
    HRESULT hr = pVoice -> Speak(text_w,SPF_ASYNC | (purge?SPF_PURGEBEFORESPEAK:0),NULL);

    if (FAILED(hr)) {
      _cprintf("Speech output error on device\n",cur_device);
      return -1;
    }
    
  }

  return 0;

}


int CMultiSpeaker::speak(int device_mask, const char* text, int purge) {

  if (initialized == 0 || text == 0) return -1;

  int num_chars = strlen(text);
  
  // I don't claim to _really_ understand wide-character strings,
  // so I always give myself a little room to mess up.
  unsigned short* speechbuf = new unsigned short[num_chars+5];

  // Copy the text that we're supposed to speak to a wide-character string
  memset(speechbuf,0,num_chars*sizeof(unsigned short));
  mbstowcs(speechbuf, text, num_chars+1);

  // Send out the wide-character buffer
  int result = speak_w(device_mask,speechbuf,purge);

  delete [] speechbuf;
  
  return result;
}

#endif