#include "stdafx.h"
#include "multidev_mp3.h"
#include <strsafe.h>

// Disable annoying STL warnings about long debug symbol names
#pragma warning(disable:4786)

#include <dshow.h>
#include <conio.h>

#include <list>
using std::list;

#include <string>
using std::string;

#include <map>
using std::map;

// Every time someone requests a new graph, it will be 
// associated with a handle, guaranteed to be unique up
// to 32 bits.
unsigned long g_current_graph_handle = 0;

#define ONE_SECOND 10000000
#define SAFE_RELEASE(x) { if (x) {(x)->Release(); x = NULL;}; }

typedef list<IMoniker*> moniker_ptr_list;
moniker_ptr_list g_hardware_monikers;

typedef map<int,IGraphBuilder*> graph_ptr_map;
graph_ptr_map g_graph_map;

int g_dshow_initialized = 0;

// A list of graph handles that need to be deleted
list<unsigned long> g_graphs_to_delete;

/***
  Adds a filter with the specified classid to the specified graph.
  Doesn't connect him to anyone.

  More or less taken from the MSDN samples.
***/
HRESULT AddFilterByCLSID(
  IGraphBuilder *pGraph,  // Pointer to the Filter Graph Manager.
  const GUID& clsid,      // CLSID of the filter to create.
  LPCWSTR wszName,        // A name for the filter.
  IBaseFilter **ppF);      // Receives a pointer to the filter.
  

/***
  Finds an unconnected pin - input or output, as specified - on
  the given filter.  Returns E_FAIL and doesn't set ppPin if no
  unconnected pin is found.

  More or less taken from the MSDN samples.
***/
HRESULT GetUnconnectedPin(
  IBaseFilter *pFilter,   // Pointer to the filter.
  PIN_DIRECTION PinDir,   // Direction of the pin to find.
  IPin **ppPin);           // Receives a pointer to the pin.

/***
  Routes a graph to the specified physical hardware devices... each
  bit in 'device_mask' represents one device.  If bit 0 is high, device 0
  is included in the graph, etc.

  Assumes the graph has exactly one renderer in it.

  Nothing bad will happen if you specify non-existant devices.
***/
HRESULT route_to_devices(IGraphBuilder* pGraph, unsigned int device_mask);
       


/***
  Enumerates all available sound devices (non-DirectSound hardware devices)
  and stores monikers for them in g_hardware_monikers.
***/
HRESULT enumerate_sound_devices();

/***
  Handle all events for an individual graph.
***/

HRESULT handle_graph_events(unsigned long graph_id, IGraphBuilder* pGraph);

/***
  Seeks the specified graph to the specified time in seconds.
***/
HRESULT seek_graph(IGraphBuilder* pGraph, double seconds, int operation);

/***
  Set the volume of the specified graph to the specified volume (should range
  from 0 to 1).
***/
HRESULT set_graph_volume(IGraphBuilder* pGraph, double volume);
  
/***
  Set the pan of the specified graph to the specified volume (should range
  from -1.0 to 1.0).
***/
HRESULT set_graph_pan(IGraphBuilder* pGraph, double pan);


// A debugging method to print all pin names on a filter
void enumerate_pins(IBaseFilter* pFilter);

int media_control(int graph_handle, int operation, double parameter) {
  
  if (g_dshow_initialized == 0) {
    _cprintf("Dshow not initialized\n");
    return -1;
  }

  // Find a handle to this graph
  graph_ptr_map::iterator graph_iter = g_graph_map.find(graph_handle);
  if (graph_iter == g_graph_map.end()) {
    _cprintf("Could not find handle %d\n",graph_handle);
    return ERROR_GRAPH_NOT_FOUND;
  }

  IGraphBuilder* pGraph = (*graph_iter).second;

  // Get seeking and media control handles to this media
  
  // Get a handle to a seeking interface for this graph
  IMediaSeeking *pSeek = NULL;
  HRESULT hr = pGraph->QueryInterface(IID_IMediaSeeking, (void **)&pSeek);

  if (FAILED(hr)) {
    _cprintf("Could not query for seeking interface\n");
    return -1;
  }
  
  // Get a media control handle for this graph
  IMediaControl *pControl = NULL;
  hr = pGraph->QueryInterface(IID_IMediaControl, (void **)&pControl);

  if (FAILED(hr)) {
    _cprintf("Could not get graph control interface...\n");
    pSeek->Release();
    return -1;
  }
  
  int toReturn = 0;

  switch(operation) {

  case MEDIA_CONTROL_NULL :

    break;

  case MEDIA_CONTROL_PLAY :

    hr = pControl->Run();

    if (FAILED(hr)) {
      _cprintf("Error running graph %d\n",graph_handle);
      toReturn = -1;
    }

    break;

  case MEDIA_CONTROL_STOP :

    hr = pControl->Stop();

    if (FAILED(hr)) {
      _cprintf("Error stopping graph %d\n",graph_handle);
      toReturn = -1;
    }

    pGraph->Release();
    _cprintf("Removing a graph from the map after a stop request...\n");
    g_graph_map.erase(graph_handle);

    break;

  case MEDIA_CONTROL_PAUSE :

    hr = pControl->Pause();

    if (FAILED(hr)) {
      _cprintf("Error pausing graph %d\n",graph_handle);
      toReturn = -1;
    }

    break;

  case MEDIA_CONTROL_SEEK :
    
    hr = seek_graph(pGraph,parameter,operation);

    if (FAILED(hr)) {
      _cprintf("Error seeking graph %d\n",graph_handle);
      toReturn = -1;
    }

    break;

  case MEDIA_CONTROL_SKIP :

    hr = seek_graph(pGraph,parameter,operation);

    if (FAILED(hr)) {
      _cprintf("Error skipping graph %d\n",graph_handle);
      toReturn = -1;
    }
  
    break;

  case MEDIA_CONTROL_SET_STOP_POSITION :

    hr = seek_graph(pGraph,parameter,operation);

    if (FAILED(hr)) {
      _cprintf("Error seeking graph %d\n",graph_handle);
      toReturn = -1;
    }

    break;

  case MEDIA_CONTROL_SET_VOLUME :

    hr = set_graph_volume(pGraph,parameter);

    if (FAILED(hr)) {
      _cprintf("Error setting graph %d volume\n",graph_handle);
      toReturn = -1;
    }

    break;

  case MEDIA_CONTROL_SET_PAN :

    hr = set_graph_pan(pGraph,parameter);

    if (FAILED(hr)) {
      _cprintf("Error setting graph %d pan\n",graph_handle);
      toReturn = -1;
    }

    break;

  default:
    _cprintf("Unrecognized operation\n");
    toReturn = -1;

  }

  pSeek->Release();
  pControl->Release();

  return toReturn;
}


int dshow_initialized() {
  return g_dshow_initialized;
}


int init_dshow() {

  if (g_dshow_initialized) return 0;

  // Things that only need to be done once...

  // Initialize the COM library.
  HRESULT hr = CoInitialize(NULL);

  if (FAILED(hr)) {
    _cprintf("Error: could not initialize COM");
    return -1;
  }

  // Enumerate available sound devices into the global list
  hr = enumerate_sound_devices();
  
  if (FAILED(hr)) {
    _cprintf("Error: could not enumerate sound devices");
    return -1;
  }
  
  g_dshow_initialized = 1;

  return 0;
}


int cleanup_dshow() {

  _cprintf("Cleaning up direct show...\n");

  // Release all unreleased graphs
  graph_ptr_map::iterator map_iter;
  for(map_iter = g_graph_map.begin();
      map_iter != g_graph_map.end();
      map_iter++) {

     IGraphBuilder* cur_graph = (*map_iter).second;     
     cur_graph->Release();

  }

  g_graph_map.clear();
  
  // Release all unreleased monikers
  moniker_ptr_list::iterator moniker_iter;
  for(moniker_iter = g_hardware_monikers.begin();
      moniker_iter != g_hardware_monikers.end();
      moniker_iter++) {

     IMoniker* cur_moniker = (*moniker_iter);
     cur_moniker->Release();

  }

  g_hardware_monikers.clear();

  // Release COM
  CoUninitialize();

  return 0;
}


HRESULT GetUnconnectedPin(IBaseFilter *pFilter,PIN_DIRECTION PinDir,IPin **ppPin) {

  *ppPin = 0;
  IEnumPins *pEnum = 0;
  IPin *pPin = 0;
  HRESULT hr = pFilter->EnumPins(&pEnum);

  if (FAILED(hr)) {
    return hr;
  }
  
  while (pEnum->Next(1, &pPin, NULL) == S_OK) {
    PIN_DIRECTION ThisPinDir;
    pPin->QueryDirection(&ThisPinDir);
    if (ThisPinDir == PinDir) {
      IPin *pTmp = 0;
      hr = pPin->ConnectedTo(&pTmp);

      // Already connected, not the pin we want.
      if (SUCCEEDED(hr)) {
        pTmp->Release();
      }
      // Unconnected, this is the pin we want.
      else {
        pEnum->Release();
        *ppPin = pPin;
        return S_OK;
      }
    }
    
    pPin->Release();
  }
  
  pEnum->Release();

  // Did not find a matching pin.
  return E_FAIL;
}


// A debugging method that prints pin names...
void enumerate_pins(IBaseFilter* pFilter) {

	IEnumPins  *pEnumPins = NULL;
    IPin       *pPin = NULL;

    HRESULT hr = pFilter->EnumPins(&pEnumPins);

    // Enumerate each pin
    while(pEnumPins->Next(1, &pPin, 0) == S_OK) {

      PIN_INFO pinfo;
      pPin->QueryPinInfo(&pinfo);
      pPin->Release();
      
      char szPinName[MAX_FILTER_NAME];
      int cch = WideCharToMultiByte(CP_ACP, 0, pinfo.achName,
        MAX_FILTER_NAME, szPinName, MAX_FILTER_NAME, 0, 0);
      _cprintf("Pin name is %s\n",szPinName);

      // Reference counting requires us to release this pointer...
      // isn't COM awesome?
      pinfo.pFilter->Release();
    }
    
    // Release the enumerator
    pEnumPins->Release();

}

HRESULT route_to_devices(IGraphBuilder* pGraph, unsigned int device_mask) {

  // Don't bother doing anything if no devices are requested
  if (device_mask == 0) {
    _cprintf("Warning: empty device list\n");
    return S_OK;
  }

  // An enumerator for available filters
  IEnumFilters *pEnumFilters = NULL;

  // A temporary pointer for a filter
  IBaseFilter *pFilter = 0;

  // This will point to the renderer we find in the graph
  IBaseFilter *pRenderer = 0;
  ULONG cFetched;

  // Enumerate filters in the graph
  HRESULT hr = pGraph->EnumFilters(&pEnumFilters);
  if (FAILED(hr)) {
    _cprintf("Could not enumerate filters...\n");
    return hr;
  }

  // Loop over the whole filter enumeration
  while(pEnumFilters->Next(1, &pFilter, &cFetched) == S_OK) {
        
    FILTER_INFO FilterInfo;
    hr = pFilter->QueryFilterInfo(&FilterInfo);
    if (FAILED(hr)) {
      _cprintf("Warning: could not get filter info\n");
      continue;
    }

    // Find the name of this filter
    char szName[MAX_FILTER_NAME];
    int cch = WideCharToMultiByte(CP_ACP, 0, FilterInfo.achName,
      MAX_FILTER_NAME, szName, MAX_FILTER_NAME, 0, 0);
    if (cch > 0) {
      // _cprintf("Filter name is %s\n",szName);
    }
    else {
      _cprintf("Warning: could not get filter name\n");
    }
            
    // Let's figure out if this filter is a renderer...
    int is_renderer = 1;
    
    IEnumPins  *pEnumPins = NULL;
    IPin       *pPin = NULL;

    // Enumerate all his pins... an output pin means he's not a
    // renderer.
    hr = pFilter->EnumPins(&pEnumPins);

    // Enumerate each pin
    while(pEnumPins->Next(1, &pPin, 0) == S_OK) {

      PIN_INFO pinfo;
      pPin->QueryPinInfo(&pinfo);
      pPin->Release();
      
      char szPinName[MAX_FILTER_NAME];
      int cch = WideCharToMultiByte(CP_ACP, 0, pinfo.achName,
        MAX_FILTER_NAME, szPinName, MAX_FILTER_NAME, 0, 0);
      //_cprintf("Pin name is %s\n",szPinName);

      if (pinfo.dir == PINDIR_OUTPUT) {
        is_renderer = 0;
        break;
      }

      else {
        // Do nothing for an input pin... just keep looking.
      }      

      // Reference counting requires us to release this pointer...
      // isn't COM awesome?
      pinfo.pFilter->Release();
    }
    
    // Release the enumerator
    pEnumPins->Release();

    // Okay, this filter is a renderer...
    if (is_renderer) {
      //_cprintf("This is a renderer...\n");
      pRenderer = pFilter;      
    }
    
    // Release the filter itself
    pFilter->Release();
    
    // The FILTER_INFO structure holds a pointer to the Filter Graph
    // Manager, with a reference count that must be released.
    if (FilterInfo.pGraph != NULL) {
      FilterInfo.pGraph->Release();
    }       

    // Stop looping if we found a renderer...
    if (pRenderer != 0) break;
  }
    
  // Release the filter enumeration
  pEnumFilters->Release();
  
  if (pRenderer == 0) {
    _cprintf("Could not find a renderer...\n");
    return E_FAIL;
  }

  // Find renderer input
  IEnumPins* pEnumPins;
	pRenderer->EnumPins(&pEnumPins);

  // renderer input pin
  IPin*	renderer_input_pin;
	
  // Enumerate input pins on this renderer
  pEnumPins->Reset();

  // We assume he has just one input, so this is it...
	hr = pEnumPins->Next(1, &renderer_input_pin, 0);

  pEnumPins->Release();

  if (FAILED(hr)) {
    _cprintf("Couldn't find renderer input pin\n");
    return hr;
  }
	
	// Find ouput pin on filter it is connected to
  IPin*	decoder_output_pin;	
  
	hr = renderer_input_pin->ConnectedTo(&decoder_output_pin);

  if (hr != S_OK) {
    _cprintf("Could not find decoder output pin...\n");
    renderer_input_pin->Release();
    return hr;
  }

  // Disconnect the filters - note that we have to call Disconnect for both pins
	hr = pGraph->Disconnect(renderer_input_pin);
	HRESULT hr2 = pGraph->Disconnect(decoder_output_pin);

  // Make sure we disconnected the filters okay...
  if (hr2 != S_OK || hr != S_OK) {
    _cprintf("Could not disconnect filter pins...\n");
    decoder_output_pin->Release();
    renderer_input_pin->Release();
    return hr;
  }
  
  // Find the inifinite tee filter by class id
  IBaseFilter* tee_filter;
  hr = AddFilterByCLSID(pGraph,CLSID_InfTee,L"infinite_tee",&tee_filter);

  if (FAILED(hr)) {
    _cprintf("Could not find infinite tee...\n");
    decoder_output_pin->Release();
    renderer_input_pin->Release();
    return hr;
  }
    
  // Find its input pin and first output pin
	IPin* tee_input_pin = 0;
  hr = GetUnconnectedPin(tee_filter,PINDIR_INPUT,&tee_input_pin);

  if (FAILED(hr)) {
    _cprintf("Warning: could not find tee input pin\n");
    tee_filter->Release();
    decoder_output_pin->Release();
    renderer_input_pin->Release();
    return hr;
  }

  // now connect decoder output to tee input
	hr = pGraph->Connect(decoder_output_pin, tee_input_pin);

  if (FAILED(hr)) {
    _cprintf("Warning: could not connect decoder pins to tee input\n");
    tee_filter->Release();
    decoder_output_pin->Release();
    renderer_input_pin->Release();
    return hr;
  }
	
  // Now build filters for each requested device from the stored
  // monikers...
  moniker_ptr_list::iterator moniker_iter;

  // Loop over all available hardware filters...
  int i;
  for(moniker_iter = g_hardware_monikers.begin(), i=0;
      moniker_iter != g_hardware_monikers.end();
      moniker_iter++,i++) {

    // Was this device requested?
    if ((device_mask & (1<<i)) == 0) {
      //_cprintf("Passing on device %x, %x\n",i,device_mask);
      continue;
    }

    // The rendering device itself
    IBaseFilter *pFilter;

    // The previously-stored moniker for this device
    IMoniker* pMoniker = (*moniker_iter);

    // Create a filter for this device
    hr = pMoniker->BindToObject(NULL, NULL, IID_IBaseFilter,
                (void**)&pFilter);

    // I thought I would have to release this pointer, but everything
    // goes to hell when I release this pointer.
    // pMoniker->Release();
      
    if (FAILED(hr)) {
      _cprintf("Could not bind moniker to filter (%d)\n",i);
      continue;
    }

    // Create a friendly name for this filter, so we can keep track of it 
    // later if we're debugging
    char name[100];
    StringCchPrintfA(name,100,"Filter %d",i);
    unsigned short wname[200];
    mbstowcs(wname,name,strlen(name)+1);

    // Add this filter to our graph
    hr = pGraph->AddFilter(pFilter,wname);

    pFilter->Release();

    if (FAILED(hr)) {
      _cprintf("Could not add filter to graph (%d)\n",i);      
      continue;
    }
    
    // Find the current tee output pin
    IPin* tee_output_pin;
  	hr = GetUnconnectedPin(tee_filter,PINDIR_OUTPUT,&tee_output_pin);

    if (FAILED(hr)) {
      _cprintf("Could not get tee output pin (%d)\n",i);
      continue;
    }

    // Grab the input pin on our device
    IPin* device_input_pin;
  	hr = GetUnconnectedPin(pFilter,PINDIR_INPUT,&device_input_pin);

    if (FAILED(hr)) {
      _cprintf("Could not get device input pin (%d)\n",i);
      tee_output_pin->Release();
      continue;
    }

    // Connect the tee to the device
    hr = pGraph->Connect(tee_output_pin,device_input_pin);

    tee_output_pin->Release();
    device_input_pin->Release();

    if (FAILED(hr)) {
      _cprintf("Could not connect tee to device (%d)\n",i);
      continue;
    }
  }

  return S_OK;  

}


#define force_default_env_variable "MULTISND_FORCE_DEFAULT"

HRESULT enumerate_sound_devices() {

  // Find out whether we need to include the default dsound device
  // as device 0...
  int use_default_dsound_as_dev0 = 0;

  char* result = getenv(force_default_env_variable);

  if (result != 0) {
    use_default_dsound_as_dev0 = 1;
  }

  if (use_default_dsound_as_dev0) {
    _cprintf("Using default dsound device as dev 0\n\n",result);
  }
   
  // Create the System Device Enumerator.
  HRESULT hr;
  ICreateDevEnum *pSysDevEnum = NULL;
  hr = CoCreateInstance(CLSID_SystemDeviceEnum, NULL, CLSCTX_INPROC_SERVER,
    IID_ICreateDevEnum, (void **)&pSysDevEnum);

  if (FAILED(hr)) {
    _cprintf("Could not create system device enumerator\n");
    return hr;
  }

  // Obtain a class enumerator for the audio renderer category.
  IEnumMoniker *pEnumCat = NULL;
  hr = pSysDevEnum->CreateClassEnumerator(CLSID_AudioRendererCategory, &pEnumCat, 0);

  // Can't use the normal FAILED() macro here...
  if (hr != S_OK) {
    _cprintf("Could not create class enumerator\n");
    pSysDevEnum->Release();
    return hr;
  }

  // Enumerate the monikers for available devices
  IMoniker *pMoniker = NULL;
  ULONG cFetched;

  // Put all the dsound device pointers in a temporary list, so we can
  // re-sort them later
  moniker_ptr_list temp_hardware_monikers;

  // A list of "real" hardware device names, in
  // "real" (media sdk) order
  list<string> ordered_hardware_name_list;

  // A list of the directsound device names, in the
  // order in which they're enumerated.
  list<string> dsound_name_list;

  // For each available device...
  while(pEnumCat->Next(1, &pMoniker, &cFetched) == S_OK) {
  
    // Bind to storage to get useful properties
    IPropertyBag *pPropBag;

    hr = pMoniker->BindToStorage(0, 0, IID_IPropertyBag, (void **)&pPropBag);

    if (FAILED(hr)) {
      _cprintf("Warning: could not bind a device to storage...\n");
      char buf[1000];
      AMGetErrorText(hr,buf,1000);
      _cprintf("Error: %s\n",buf);
      pMoniker->Release();
      continue;
    }

    // Retrieve the filter's friendly name
    VARIANT varName;
    VariantInit(&varName);
    
    hr = pPropBag->Read(L"FriendlyName", &varName, 0);
        
    /*

    // If we were using IPropertyBag2, we would do...
    PROPBAG2 pb2;
    pb2.vt = VT_BSTR;
    pb2.pstrName = L"FriendlyName";
    pb2.cfType = 0;
    pb2.dwHint = 0;
    
    HRESULT hr_read = pPropBag->Read(1,&pb2,0,&varName,&hr);

    */
   
    char friendly_name[1000];
      
    if (FAILED(hr)) {      
      _cprintf("Failed to get device friendly name\n");
      pMoniker->Release();
      pPropBag->Release();
      continue;
    }

    wcstombs(friendly_name,varName.bstrVal,wcslen(varName.bstrVal)+1);

    _cprintf("Device friendly name is %s\n",friendly_name);

    // Need to find out whether this is really a hardware
    // device...
    int is_hardware_device = 1;
    
    // TODO: Find a better way to do this.  I experimented with
    // using IPropertyBag2 so I could enumerate the available 
    // properties, but that didn't work (because it didn't
    // actually subclass from IPropertyBag).  And IPropertyBag
    // doesn't allow enumeration.

    
    /* 
       So I find that the devices called "DirectSound:" can mix multiple
       mp3's, and the others can't.  I don't at all understand the order
       in which these devices appear; it doesn't seem to be related to the
       ordering in the sound control panel.  But it works, and that's what
       counts.
    */

    if (strstr(friendly_name,"DirectSound:") == 0) is_hardware_device = 0;
    
    // I'm now getting my "ordered hardware name list" from the waveout
    // interface...

    /*
    // ...but maybe it's the _name_ of a hardware device
    if (
         (strstr(friendly_name,"DirectSound") == 0) 
         &&
         (strstr(friendly_name,"Default") == 0)
         ) {
      _cprintf("Found a hardware _name_ : %s\n",friendly_name);
      hardware_name_list.push_back((string)(friendly_name));
    }
    */
    
    if (is_hardware_device) {

      // _cprintf("Found a hardware device: %s\n",friendly_name);
      temp_hardware_monikers.push_back(pMoniker);
      dsound_name_list.push_back((string)(friendly_name));

      // Enumerate the pins on this device for debugging...
      
      /*
      IBaseFilter* pFilter;

      // Create a filter for this device
      pMoniker->BindToObject(NULL, NULL, IID_IBaseFilter,(void**)&pFilter);

      enumerate_pins(pFilter);

      pFilter->Release();
      */

    }

    // If we're supposed to force the default device onto the list...
    else if (use_default_dsound_as_dev0 &&
      (strstr(friendly_name,"Default") != 0) &&
      (strstr(friendly_name,"DirectSound") != 0) ) {

      g_hardware_monikers.push_back(pMoniker);

    }
    
    

    pPropBag->Release();
    
    VariantClear(&varName);         
    
  }
  
  _cprintf("\n");

  // Now collect hardware names in "real name" order
  
  WAVEOUTCAPS caps;
  unsigned int num_devices = waveOutGetNumDevs();;

  for(unsigned int i  = 0; i < num_devices; i++) {
    waveOutGetDevCaps(i,&caps,sizeof(WAVEOUTCAPS));
    _cprintf("WAV device %02d: %s\n",i,caps.szPname);
    ordered_hardware_name_list.push_back(caps.szPname);
  }

  _cprintf("\n");

  std::list<string>::iterator name_iter;

  // Print out the directsound devices for debugging, before ordering
  for(name_iter = dsound_name_list.begin();
      name_iter != dsound_name_list.end();
      name_iter++) {
      const char* dsound_name = (*name_iter).c_str();
      _cprintf("Directsound device: %s\n",dsound_name);
  }

  _cprintf("\n");

  // Now move monikers to the real list in "real name" order
  for(name_iter = ordered_hardware_name_list.begin();
      name_iter != ordered_hardware_name_list.end();
      name_iter++) {

    const char* hardware_name = (*name_iter).c_str();

    // Loop over all devices to see whether they match...
    moniker_ptr_list::iterator moniker_iter = temp_hardware_monikers.begin();
    std::list<string>::iterator dsound_name_iter = dsound_name_list.begin();

    for(; dsound_name_iter != dsound_name_list.end(); dsound_name_iter++) {

      const char* dsound_name = (*dsound_name_iter).c_str();

      // If we found a match...
      if (strstr(dsound_name,hardware_name) != 0) {
        _cprintf("%s matches %s...\n",dsound_name,hardware_name);
        g_hardware_monikers.push_back((*moniker_iter));

        temp_hardware_monikers.erase(moniker_iter);
        dsound_name_list.erase(dsound_name_iter);
        break;
      }

      moniker_iter++;      
    }

  } // Looping over all hardware names

  _cprintf("\n");

  // See whether there are any dsound devices left...
  if (temp_hardware_monikers.size() != 0) {
    _cprintf("Warning: some dsound devices were not assigned ordererd names...\n");

    moniker_ptr_list::iterator moniker_iter = temp_hardware_monikers.begin();

    // Assign remaining monikers to the hardware list
    for(; moniker_iter != temp_hardware_monikers.end(); moniker_iter++) {
      g_hardware_monikers.push_back(*moniker_iter);
    }

  }

    // Now check each device and see whether it matches...
  // Release the enumeration machinery
  pEnumCat->Release();
  pSysDevEnum->Release();

  // Make sure that we found the "real" number of sound output devices
  if (g_hardware_monikers.size() != waveOutGetNumDevs()) {
    _cprintf("Warning: found %d devices, but waveout says there should be %d\n",
      g_hardware_monikers.size(), waveOutGetNumDevs());
  }

  _cprintf("\n");

  return S_OK;

}



HRESULT AddFilterByCLSID(IGraphBuilder *pGraph,const GUID& clsid,
  LPCWSTR wszName, IBaseFilter **ppF) {

  // Verify input arguments
  if (!pGraph || ! ppF) return E_POINTER;

  *ppF = 0;
  IBaseFilter *pF = 0;

  // Create the filter by class ID
  HRESULT hr = CoCreateInstance(clsid, 0, CLSCTX_INPROC_SERVER,
      IID_IBaseFilter, reinterpret_cast<void**>(&pF));

  if (FAILED(hr)) {
    _cprintf("Error creating filter %s\n",wszName);
    return hr;
  }

  // Add the filter to the graph if it was successfully created
  hr = pGraph->AddFilter(pF, wszName);

  if (FAILED(hr)) {
    _cprintf("Error adding filter %s\n",wszName);
    pF->Release();
    return hr;
  }
     
  // Set the return argument
  *ppF = pF;
  char buf[1000];
  wcstombs(buf,wszName,wcslen(wszName)+1);
  //_cprintf("Added filter %s\n",buf);
  
  return hr;

}


int get_graph_status(IGraphBuilder* pGraph) {

  HRESULT hr;

  // Get a media control handle for this graph
  IMediaControl *pControl = NULL;
  hr = pGraph->QueryInterface(IID_IMediaControl, (void **)&pControl);

  if (FAILED(hr)) {
    _cprintf("Could not get graph control interface for checking status...\n");      
    return -1;
  }

  OAFilterState state;
  hr = pControl->GetState(50,&state);

  if (FAILED(hr)) {
    _cprintf("Could not get graph state\n");
    pControl->Release();
    return hr;
  }

  pControl->Release();

  return state;
    
}

int handle_all_graph_events() {

  // Check for events on each available graph...
  graph_ptr_map::iterator map_iter;

  for(map_iter = g_graph_map.begin();
      map_iter != g_graph_map.end();
      map_iter++) {

    IGraphBuilder* pGraph = (*map_iter).second;
    unsigned long id = (*map_iter).first;
    handle_graph_events(id, pGraph);
  }

  // See whether any graphs should be removed...
  std::list<unsigned long>::iterator deletion_iter;

  for(deletion_iter  = g_graphs_to_delete.begin();
      deletion_iter != g_graphs_to_delete.end();
      deletion_iter++) {

     unsigned long graph_id = (*deletion_iter);

     map_iter = g_graph_map.find(graph_id);

     if (map_iter == g_graph_map.end()) {
       _cprintf("Warning: could not find graph %d for deletion\n");
     }

     else {
       IGraphBuilder* pGraph = (*map_iter).second;
       g_graph_map.erase(map_iter);
       _cprintf("Deleting graph %d\n",graph_id);
       pGraph->Release();
    }
  }

  g_graphs_to_delete.clear();
  
  return 0;
}

HRESULT handle_graph_events(unsigned long graph_id, IGraphBuilder* pGraph) {

  // Disregard if we don't have an IMediaEventEx pointer.
  if (pGraph == NULL) {
    _cprintf("Warning: no graph available\n");
    return E_FAIL;
  }
  
  // Get an event interface.  Normally I would do this once, ahead
  // of time, but having multiple graphs is easier if I just _always_
  // use the graph pointer.
  IMediaEventEx* pEvent;
  HRESULT hr = pGraph->QueryInterface(IID_IMediaEventEx, (void **)&pEvent);

  if (FAILED(hr)) {
    _cprintf("Could not get an event interface to handle events...\n");
    return hr;
  }
  
  // Get all the events
  long evCode;
  long param1, param2;
  
  while (SUCCEEDED(pEvent->GetEvent(&evCode, &param1, &param2, 0))) {
        
    // Even this little struct needs to be released.  Yey com.
    pEvent->FreeEventParams(evCode, param1, param2);

    switch (evCode) {
      case EC_COMPLETE:
        _cprintf("Playback complete...\n");

        // Mark this graph for deletion
        g_graphs_to_delete.push_back(graph_id);

        break;

      case EC_USERABORT:
        _cprintf("Playback aborted...\n");
        break;
      case EC_ERRORABORT:
        _cprintf("Playback error...\n");
        break;
    }

  } // Looping over all pending events

  pEvent->Release();

  return S_OK;

} // handle_graph_events()


double get_duration(int graph_handle) {

  // Make sure dshow is initialized
  if (g_dshow_initialized == 0) {
    _cprintf("Dshow not initialized\n");
    return (double)(-1);
  }

  // Find a handle to this graph
  graph_ptr_map::iterator graph_iter = g_graph_map.find(graph_handle);
  if (graph_iter == g_graph_map.end()) {
    _cprintf("Could not find handle %d\n",graph_handle);
    return (double)ERROR_GRAPH_NOT_FOUND;
  }

  IGraphBuilder* pGraph = (*graph_iter).second;

  // Get seeking and media control handles to this media
  
  // Get a handle to a seeking interface for this graph
  IMediaSeeking *pSeek = NULL;
  HRESULT hr = pGraph->QueryInterface(IID_IMediaSeeking, (void **)&pSeek);

  if (FAILED(hr)) {
    _cprintf("Could not query for seeking interface\n");
    return (double)(-1);
  }

  REFERENCE_TIME duration;
  pSeek->GetDuration(&duration);
  double seconds = ((double)(duration))/((double)(ONE_SECOND));

  pSeek->Release();

  return seconds;
}


double get_current_position(int graph_handle) {

  // Make sure dshow is initialized
  if (g_dshow_initialized == 0) {
    _cprintf("Dshow not initialized\n");
    return (double)(-1);
  }

  // Find a handle to this graph
  graph_ptr_map::iterator graph_iter = g_graph_map.find(graph_handle);
  if (graph_iter == g_graph_map.end()) {
    _cprintf("Could not find handle %d\n",graph_handle);
    return (double)ERROR_GRAPH_NOT_FOUND;
  }

  IGraphBuilder* pGraph = (*graph_iter).second;

  // Get a handle to a seeking interface for this graph
  IMediaSeeking *pSeek = NULL;
  HRESULT hr = pGraph->QueryInterface(IID_IMediaSeeking, (void **)&pSeek);

  if (FAILED(hr)) {
    _cprintf("Could not query for seeking interface\n");
    return (double)(-1);
  }

  REFERENCE_TIME curtime;
  pSeek->GetCurrentPosition(&curtime);
  double seconds = ((double)(curtime))/((double)(ONE_SECOND));

  pSeek->Release();

  return seconds;
}


HRESULT set_graph_volume(IGraphBuilder* pGraph, double volume) {

  // Get a handle to the basic audio interface for this graph
  IBasicAudio *pAudio = NULL;
  HRESULT hr = pGraph->QueryInterface(IID_IBasicAudio, (void **)&pAudio);

  if (FAILED(hr)) {
    _cprintf("Could not query for audio interface\n");
    return hr;
  }

  // Clip volume to the allowed range
  if (volume < 0) volume = 0;
  if (volume > 1.0) volume = 1.0;

  // Scale volume to the DirectShow range (-10000 --> 0)
  long v = volume * 10000 - 10000;

  hr = pAudio->put_Volume(v);

  if (FAILED(hr)) {
    _cprintf("Could not set graph volume\n");
    pAudio->Release();
    return hr;
  }

  pAudio->Release();

  return S_OK;
}


HRESULT set_graph_pan(IGraphBuilder* pGraph, double pan) {

  // Get a handle to the basic audio interface for this graph
  IBasicAudio *pAudio = NULL;
  HRESULT hr = pGraph->QueryInterface(IID_IBasicAudio, (void **)&pAudio);

  if (FAILED(hr)) {
    _cprintf("Could not query for audio interface\n");
    return hr;
  }

  // Clip pan to the allowed range
  if (pan < -1.0) pan = -1.0;
  if (pan > 1.0) pan = 1.0;

  // Scale volume to the DirectShow range (-10000 --> 10000)
  long p = pan * 10000;

  hr = pAudio->put_Balance(p);

  if (FAILED(hr)) {
    _cprintf("Could not set graph pan\n");
    pAudio->Release();
    return hr;
  }

  pAudio->Release();

  return S_OK;
}


HRESULT seek_graph(IGraphBuilder* pGraph, double seconds, int operation) {
  
  // Get a handle to a seeking interface for this graph
  IMediaSeeking *pSeek = NULL;
  HRESULT hr = pGraph->QueryInterface(IID_IMediaSeeking, (void **)&pSeek);

  if (FAILED(hr)) {
    _cprintf("Could not query for seeking interface\n");
    return hr;
  }

  // Convert negative times to absolute if necessary
  if ( (seconds < 0) && (operation == MEDIA_CONTROL_SEEK || operation == MEDIA_CONTROL_SET_STOP_POSITION) ) {
      
    REFERENCE_TIME duration;
    pSeek->GetDuration(&duration);
    seconds = ((double)(duration))/((double)(ONE_SECOND)) + seconds;

  }
  
  // Convert seconds to DirectShow units
  REFERENCE_TIME new_position = (REFERENCE_TIME)(seconds*((double)(ONE_SECOND)));
  
  LONGLONG* current_position = 0;
  DWORD current_flags = AM_SEEKING_NoPositioning;

  LONGLONG* stop_position = 0;
  DWORD stop_flags = AM_SEEKING_NoPositioning;

  switch(operation) {
  case MEDIA_CONTROL_SEEK:

    current_position = &new_position;
    current_flags = AM_SEEKING_AbsolutePositioning;

    break;

  case MEDIA_CONTROL_SET_STOP_POSITION:

    stop_position = &new_position;
    stop_flags = AM_SEEKING_AbsolutePositioning;

    break;

  case MEDIA_CONTROL_SKIP:

    current_position = &new_position;
    current_flags = AM_SEEKING_RelativePositioning;

    break;

  default:
    _cprintf("Unrecognized seek operation...\n");
  }
  
  hr = pSeek->SetPositions(
    current_position,current_flags,stop_position,stop_flags 
  );
  
  if (FAILED(hr)) {
    _cprintf("Could not seek graph\n");
    pSeek->Release();
    return hr;
  }

  pSeek->Release();

  return S_OK;
}
  

int build_graph_for_file(const char* filename, int device_mask) {

  IGraphBuilder* pGraph = 0;

  HRESULT hr;

  // Something in the Diamondtouch library seems to uninitialize
  // COM sometimes.  So I excessively initialize it here.
  //
  // It doesn't seem to have any performance impact.
  CoInitialize(NULL);
  
  // Create the filter graph manager and query for interfaces.
  hr = CoCreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC_SERVER, 
                        IID_IGraphBuilder, (void **)&pGraph);

  if (FAILED(hr)) {
    _cprintf("Error: Could not create the Filter Graph Manager for file %s\n",filename);
    return -1;
  }

  // Build the graph.
  unsigned short wc_filename[1000];
  mbstowcs(wc_filename, filename, strlen(filename)+1);

  hr = pGraph->RenderFile(wc_filename, NULL);
  
  if (FAILED(hr)) {
    _cprintf("Could not create filter graph...\n");
    pGraph->Release();
    return -1;
  }

  // Route this graph to the appropriate devices
  hr = route_to_devices(pGraph,device_mask);

  if (FAILED(hr)) {
    _cprintf("Could not route graph to specific devices...\n");
    pGraph->Release();
    return -1;
  }
 
  /*
  // How to set up window notification
  IMediaEventEx* pEvent;
  hr = pGraph->QueryInterface(IID_IMediaEventEx, (void **)&pEvent);
  if (FAILED(hr)) {
    _cprintf("Could not query graph for event notification...\n");
    pGraph->Release();
    return -1;
  }

  #define WM_GRAPHNOTIFY  WM_APP + 1004
  pEvent->SetNotifyWindow((OAHWND)g_hWnd, WM_GRAPHNOTIFY, 0);
  pEvent->Release();
  */
  
  int handle = g_current_graph_handle;
  g_current_graph_handle++;

  g_graph_map[handle] = pGraph;

  _cprintf("Built a graph (handle %d) for %s\n",handle,filename);

  return handle;
}
