%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% function animate_predictions(bin_times,predict_x,predict_y,real_xpos,real_ypos,smoothing_width)
%
% Makes nice animations of predicted trajectories.  The format of the parameters
% is a little bizarre, but the point is to be compatible with the other scripts
% I'm using.  In particular, there's no good reason that the predicted and real
% positions are in totally different formats, other than the fact that they come
% from different places.
%
% bin_times : this is a vector of time (seconds) that defines the start of each
% bin that should be animated
%
% predict_x,predict_y : vectors of x/y positions, one entry per bin
%
% real_xpos, real_ypos : each of these is a two-column vector.  The first 
% column is time, the second column is the corresponding coordinate.  They
% need not be sync'ed with the bins at all, but obviously they should contain
% the same region in time covered by bin_times.
%
% smoothing_width : if specifies, a running-average filter smoothing_width bins
% wide is applied to the predicted x and y variables.
%
% Dan Morris, Stanford University, 2003
% http://techhouse.brown.edu/~dmorris
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function animate_predictions(bin_times,predict_x,predict_y,real_xpos,real_ypos,smoothing_width)

if (nargin < 5)
  fprintf(1,'\nUsage: animate_predictions(bin_times,predx,predy,realx,realy)\n\n');
  return;
end

if (nargin < 6)
  smoothing_width = -1;
end

% Apply a running-average filter to the predicted data
if (smoothing_width > 0)
  b = ones(1,smoothing_width);
  a = [smoothing_width];
  predict_x = filter(b,a,predict_x);
  predict_y = filter(b,a,predict_y);
end

% close open graphs
% close all;

start_time = bin_times(1);
cur_bin = 1;
cur_x_index = 1;
cur_y_index = 1;

% find the indices in the 'real' x and y arrays corresponding to the first
% time point
while(cur_x_index <= size(real_xpos,1) & real_xpos(cur_x_index,1) < start_time)
  cur_x_index = cur_x_index + 1;
end

while(cur_y_index <= size(real_ypos,1) & real_ypos(cur_y_index,1) < start_time)
  cur_y_index = cur_y_index + 1;
end

h = figure(gcf+1);
set(h,'Name','Filter animation');
set(h,'MenuBar','none');
xlabel('X (tablet units)');
ylabel('Y (tablet units)');
set(h,'DoubleBuffer','on');

% plot the first real and predicted points, and get handles to them so we
% can just move the data around later without re-drawing the plot
predict_handle = plot(predict_x(1),predict_y(1),'x');
hold on;
real_handle = plot(real_xpos(cur_x_index,2),real_ypos(cur_y_index,2),'o');
title('Filter animation: the circle represents hand position, the ''x'' represents predictions');

% Find the maximum and minimum values to set nice axis limits
minx = min(real_xpos(:,2));
minx = min([minx min(predict_x)]);

miny = min(real_ypos(:,2));
miny = min([miny min(predict_y)]);

maxx = max(real_xpos(:,2));
maxx = max([maxx max(predict_x)]);

maxy = max(real_ypos(:,2));
maxy = max([maxy max(predict_y)]);

max_axis_size = max([(maxx - minx) (maxy - miny)]);

% set nice axis limits
axis([minx minx+max_axis_size miny miny+max_axis_size]);

% Record the time for the start of frame 0
tic;

start_cpu_time = toc;

while(cur_bin <= length(bin_times))

  cur_cpu_time = toc;

  % what time are we at in our animation
  cur_plex_time = (cur_cpu_time - start_cpu_time) + start_time;

  % advance to the current bin in the predictions
  while(cur_bin <= length(bin_times) & bin_times(cur_bin) < cur_plex_time)
    cur_bin = cur_bin + 1;
  end

  % advance to the current bin in the real x and y arrays
  while(cur_x_index <= size(real_xpos,1) & real_xpos(cur_x_index,1) < cur_plex_time)
    cur_x_index = cur_x_index + 1;
  end

  while(cur_y_index <= size(real_ypos,1) & real_ypos(cur_y_index,1) < cur_plex_time)
    cur_y_index = cur_y_index + 1;
  end

  % update the plots to reflect the new data
  cur_predict_x = predict_x(cur_bin);
  cur_predict_y = predict_y(cur_bin);
  cur_real_x = real_xpos(cur_x_index,2);
  cur_real_y = real_ypos(cur_y_index,2);

  set(predict_handle,'XData',[cur_predict_x]);
  set(predict_handle,'YData',[cur_predict_y]);
  set(real_handle,'XData',[cur_real_x]);
  set(real_handle,'YData',[cur_real_y]);
  drawnow
end
