%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Demonstrates the linear decoding filter... loads data, builds
% a filter on some of it, then generates and plots predictions on
% another chunk of data.
% 
% Dan Morris, Stanford University, 2003
% http://techhouse.brown.edu/~dmorris
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% GLOBAL CONSTANTS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Should we generate animations in addition to static plots?
make_animations = 1;
DATA_FILENAME = 'data/synthetic.mat';

% Define the beginning and end of the training/test periods in the
% experiment (seconds)
TRAINING_PERIOD_START = 1;
TRAINING_PERIOD_END   = 60;
TEST_PERIOD_START     = 61;
TEST_PERIOD_END       = 120;

% Throw away any spikes with fewer cells than this...
MINIMUM_SPIKES_PER_CELL = 500;

% The linear filter bin size (in seconds) and filter length (in bins)
BIN_SIZE = 0.025;
FILTER_LENGTH = 10;

% The width (samples) of the boxcar we'll apply to the predictions...
SMOOTHING_WIDTH = 15;

% These are the globals I set when working with real data...
%
% DATA_FILENAME = 'data/S020501C_CENTER_OUT_B.mat';
% TRAINING_PERIOD_START = 30;
% TRAINING_PERIOD_END   = 225;
% TEST_PERIOD_START     = 230;
% TEST_PERIOD_END       = 400;
% MINIMUM_SPIKES_PER_CELL = 800;
% FILTER_LENGTH         = 20;
% BIN_SIZE             = 0.05;
% SMOOTHING_WIDTH = 8;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fprintf(1,'\nLoading neural and kinematic data...\n');
load(DATA_FILENAME);

fprintf(1,'\nFinished loading data... building filter...\n');
f = make_linear_filter(experiment,[],BIN_SIZE,FILTER_LENGTH,TRAINING_PERIOD_START,TRAINING_PERIOD_END,MINIMUM_SPIKES_PER_CELL);

fprintf(1,'\nFinished building filter... generating predictions (on _different_ data)...\n');
[predict_x,predict_y,t] = make_predictions(experiment,f,TEST_PERIOD_START,TEST_PERIOD_END);

% Apply a running-average filter to the predicted data
if (SMOOTHING_WIDTH > 0)

  % Matlab 6 doesn't have this function, sadly...  
  % predict_x = smooth(predict_x,SMOOTHING_WIDTH);
  % predict_y = smooth(predict_y,SMOOTHING_WIDTH);

  % So we do it manually...
  b = ones(1,SMOOTHING_WIDTH);
  a = [SMOOTHING_WIDTH];
  predict_x = filter(b,a,predict_x);
  predict_y = filter(b,a,predict_y);
  w = floor(SMOOTHING_WIDTH/2);
  lastx = predict_x(end);
  lasty = predict_y(end);
  predict_x = [predict_x(w+1:end); ones(w,1)*lastx];
  predict_y = [predict_y(w+1:end); ones(w,1)*lasty];

end

% See how far off we were...
[mse, real_x_binned, real_y_binned] = prediction_error(t,predict_x,predict_y,experiment.xpos,experiment.ypos);

fprintf(1,'\nFinished generating predictions... plotting results...\n');

close all;

h = figure(1);
plot(t,predict_x,'r',t,real_x_binned,'g');
legend('Predicted x','Real x');
title('Real X (green) and predicted X (red) vs. time');
set(h,'Name','Real X (green) and predicted X (red) vs. time');
set(h,'MenuBar','none');
xlabel('Time (s)');
ylabel('Position');
zoom xon;

h = figure(2);
plot(t,predict_y,'r',t,real_y_binned,'g');
legend('Predicted y','Real y');
title('Real Y (green) and predicted Y (red) vs. time');
set(h,'Name','Real Y (green) and predicted Y (red) vs. time');
set(h,'MenuBar','none');
xlabel('Time (s)');
ylabel('Position');
zoom xon;

if (make_animations)
    fprintf(1,'\nStarting animation...\n');
    animate_predictions(t,predict_x,predict_y,experiment.xpos,experiment.ypos,9);
    return;
end

return;

% In case you want to plot velocity...

x_diff = [0; diff(real_x_binned)];
y_diff = [0; diff(real_y_binned)];
vel = sqrt(x_diff.^2 + y_diff.^2);

figure(4);
plot(vel,sqrt((predict_x-real_x_binned).^2),'r.',vel,sqrt((predict_y-real_y_binned).^2),'g.');
title('Y error (green) and X error (red) vs. velocity');
zoom xon;
