%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% function animate_invkin(base_pos,joint_predictions,smoothing_width,
%   cartesian_predictions,actual_positions);
%
% Renders a movie of a 2-joint arm, driven by the specified positions,
% along with (optionally) two moving cursors, intended to indicate
% the cartesian-space predictions and the actual hand positions that match
% the joint-space predictions represented by the arm.
%
% Can be used to generate an avi file also, if the GENERATE_MOVIE variable
% is set to '1'.
% 
% base_pos is a 2x1 vector indicating the (x,y) position of the 'shoulder'
%
% joint_predictions is a [num_predictions x 5] matrix, where each row is of
% the form:
%
% time hand_x hand_y elbow_x elbow_y
%
% All subsequent parameters are optional...
%
% smoothing_width dictates the size of the boxcar smoothing filter applied
% to the cartesian and joint-space predictions, in samples
%
% cartesian_predictions is a [num_predictions x 3] matrix, where each row is
% of the form:
%
% time predicted_x predicted_y
%
% actual_positions is a [num_positions x 3] matrix, where each row is of
% the form:
%
% time hand_position_x hand_position_y
%
% Dan Morris, Stanford University, 2003
% http://techhouse.brown.edu/~dmorris
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function animate_invkin(base_pos,joint_predictions,smoothing_width,cartesian_predictions,actual_positions);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% GLOBAL DEFINITIONS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Used to generate a movie from the animation...
GENERATE_MOVIE = 0;
MOVIE_FILENAME = 'jointspace_filtering.avi';
MOVIE_LENGTH = -1;
MOVIE_FRAME_RATE = 20;

% I use the Techsmith Screen-Capture Codec, an excellent lossless codec.
% If you don't have that installed, 'cinepak' works well enough.  Sadly,
% Matlab doesn't seem to like divx.
MOVIE_CODEC_NAME = 'TSCC';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if (nargin < 2)
    fprintf(1,'\nanimate_invkin(base_pos,joint_predictions\n');
    return;
end

x_base = base_pos(1);
y_base = base_pos(2);

joint_t = joint_predictions(:,1);
elbow_x = joint_predictions(:,2);
elbow_y = joint_predictions(:,3);
end_x = joint_predictions(:,4);
end_y = joint_predictions(:,5);

if (nargin < 3)
  smoothing_width = -1;
end

if (nargin < 4)
  cartesian_predict_t = [];
  cartesian_predict_x = [];
  cartesian_predict_y = [];
else
  cartesian_predict_t = cartesian_predictions(:,1);
  cartesian_predict_x = cartesian_predictions(:,2);
  cartesian_predict_y = cartesian_predictions(:,3);
end

if (nargin < 5)
  actual_t = [];
  actual_x = [];
  actual_y = [];
else
  actual_t = actual_positions(:,1);
  actual_x = actual_positions(:,2);
  actual_y = actual_positions(:,3);
end

% Apply a running-average filter to the predicted data
if (smoothing_width > 0)
    b = ones(1,smoothing_width);
    a = [smoothing_width];
    
    elbow_x = filter(b,a,elbow_x);
    elbow_y = filter(b,a,elbow_y);
    end_x = filter(b,a,end_x);
    end_y = filter(b,a,end_y);
    
    if (~isempty(cartesian_predict_t))
      cartesian_predict_x = filter(b,a,cartesian_predict_x);
      cartesian_predict_y = filter(b,a,cartesian_predict_y);
    end
end

% close open graphs
% close all;
h = figure(gcf+1);
set(h,'Name','Filter animation');
set(h,'MenuBar','none');
xlabel('X (cm)');
ylabel('Y (cm)');
set(h,'DoubleBuffer','on');
title('Cartesian-space and joint-space hand predictions');

start_time = joint_t(1);
cur_joint_index = 1;
cur_cartesian_index = 1;
cur_actual_index = 1;

hold on;

component_erase_mode = 'normal';

% plot the first predicted point
x1 = [x_base elbow_x(1)];
y1 = [y_base elbow_y(1)];
h1 = line(x1,y1);
set(h1,'EraseMode',component_erase_mode);

x2 = [elbow_x(1) end_x(1)];
y2 = [elbow_y(1) end_y(1)];
h2 = line(x2,y2);
set(h2,'EraseMode',component_erase_mode);

h3 = plot(x_base,y_base,'r*','EraseMode',component_erase_mode);
h4 = plot(elbow_x(1),elbow_y(1),'r*','EraseMode',component_erase_mode');
h5 = plot(end_x(1),end_y(1),'r*','EraseMode',component_erase_mode);

if (~isempty(cartesian_predict_t))
  while( (cartesian_predict_t(cur_cartesian_index) < start_time) && ...
      (cur_cartesian_index < length(cartesian_predict_t)))
    cur_cartesian_index = cur_cartesian_index+1;
  end
  h7 = plot(cartesian_predict_x(cur_cartesian_index),cartesian_predict_y(cur_cartesian_index),...
    'g*','EraseMode',component_erase_mode);
else
  h7 = 0;
end

if (~isempty(actual_t))
  while( (actual_t(cur_actual_index) < start_time) && (cur_actual_index < length(actual_t)))
    cur_actual_index = cur_actual_index+1;
  end
  h6 = plot(actual_x(cur_actual_index),actual_y(cur_actual_index),'b*','EraseMode',...
    component_erase_mode);
else
  h6 = 0;
end

legend_handles = h5;
if (h6) legend_handles = [legend_handles h6]; end;
if (h7) legend_handles = [legend_handles h7]; end;
  
legend_names = {'Joint-space prediction'};
if (h6) legend_names = {legend_names{:} 'Cartesian-space prediction'}; end;
if (h7) legend_names = {legend_names{:} 'Actual hand position'}; end;

[LEGH,OBJH,OUTH,OUTM] = legend(legend_handles,legend_names,'Location','SouthWest');
set(OBJH,'EraseMode','none');
set(OUTH,'EraseMode','none');

% Find the maximum and minimum values to set nice axis limits
minx = min([x_base min(elbow_x) min(end_x)]);
miny = min([y_base min(elbow_y) min(end_y)]);

maxx = max([x_base max(elbow_x) max(end_x)]);
maxy = max([y_base max(elbow_y) max(end_y)]);

max_axis_size = max([(maxx - minx) (maxy - miny)]);

% set nice axis limits
axis([minx minx+max_axis_size miny miny+max_axis_size]);

if (GENERATE_MOVIE) 
  % Initialize an avi file...
  cur_movie_frame = 1;
  
  aviobj = avifile(MOVIE_FILENAME);
  aviobj.fps = MOVIE_FRAME_RATE;
  frame_time = 1 / MOVIE_FRAME_RATE;
  aviobj.compression = MOVIE_CODEC_NAME;
  aviobj.quality = 100;
  aviobj.videoname = 'Joint-space reconstruction';
  
  % Write out the first frame?
  
end

% Record the time for the start of frame 0
tic;

start_cpu_time = toc;

while(cur_joint_index < length(joint_t))

    cur_cpu_time = toc;

    % what time are we at in our animation
    if (GENERATE_MOVIE)
      % Figure out the current time based on the current frame...
      cur_plex_time = start_time + frame_time * cur_movie_frame;
      
      if ((MOVIE_LENGTH > 0) && ((cur_plex_time - start_time) > MOVIE_LENGTH)) break; end;
      cur_movie_frame = cur_movie_frame + 1;
    else
      cur_plex_time = (cur_cpu_time - start_cpu_time) + start_time;
    end
    
    % advance to the current bin in the predictions
    while(cur_joint_index < length(joint_t) & joint_t(cur_joint_index) < cur_plex_time)
        cur_joint_index = cur_joint_index + 1;
    end
    
    if (~isempty(cartesian_predict_t))
      while( (cartesian_predict_t(cur_cartesian_index) < cur_plex_time) && (cur_cartesian_index < length(cartesian_predict_t)))
        cur_cartesian_index = cur_cartesian_index+1;
      end      
    end

    if (~isempty(actual_t))
      while( (actual_t(cur_actual_index) < cur_plex_time) && (cur_actual_index < length(actual_t)))
        cur_actual_index = cur_actual_index+1;
      end
    end

    % update the plots to reflect the new data
    set(h1,'XData',[x_base,elbow_x(cur_joint_index)]);
    set(h1,'YData',[y_base,elbow_y(cur_joint_index)]);

    set(h2,'XData',[elbow_x(cur_joint_index) end_x(cur_joint_index)]);
    set(h2,'YData',[elbow_y(cur_joint_index) end_y(cur_joint_index)]);
    
    set(h4,'XData',elbow_x(cur_joint_index));
    set(h4,'YData',elbow_y(cur_joint_index));
    
    set(h5,'XData',end_x(cur_joint_index));
    set(h5,'YData',end_y(cur_joint_index));
    
    if (h6)
      set(h6,'XData',cartesian_predict_x(cur_cartesian_index));
      set(h6,'YData',cartesian_predict_y(cur_cartesian_index));
    end
    
    if (h7)
      set(h7,'XData',actual_x(cur_actual_index));
      set(h7,'YData',actual_y(cur_actual_index));
    end
    
    drawnow
    
    if (GENERATE_MOVIE)      
      % Write out a frame...
      f = getframe(gcf);
      aviobj = addframe(aviobj,f);
    end
end

if (GENERATE_MOVIE);
  aviobj = close(aviobj);
end
