%%%%%%%%%%%%%%%%
%
% inv_kin_filter.m
%
% Loads a data set containing spike data and hand position data.
%
% Generates hand-position predictions using the given spike data 
% using a cartesian-space linear filter, then using a joint-space
% linear filter, and compares the results.
%
% Dan Morris, Stanford University, 2003
% http://techhouse.brown.edu/~dmorris
%
%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% GLOBAL CONSTANTS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% pick segment lengths and a base location (this is my "monkey model")
x_base = 14;
y_base = -2.0;
a1 = 10;
a2 = a1;
elbow_up = 0;

% Should we generate an animation in addition to static plots?
ANIMATE_RESULTS = 1;

% The path where the filter functions live...
FILTER_FUNCTION_PATH = '..';

DATA_FILENAME = '../data/synthetic.mat';

% Define the beginning and end of the training/test periods in the
% experiment (seconds)
TRAINING_PERIOD_START = 1;
TRAINING_PERIOD_END   = 60;
TEST_PERIOD_START     = 61;
TEST_PERIOD_END       = 120;

% Throw away any spikes with fewer cells than this...
MINIMUM_SPIKES_PER_CELL = 500;

% The linear filter bin size (in seconds) and filter length (in bins)
BIN_SIZE = 0.025;
FILTER_LENGTH = 10;

% The width (samples) of the boxcar we'll apply to the predictions...
SMOOTHING_WIDTH = 15;

UNITS_PER_CM = (1/8);

% % These are the globals I set when working with real data...
% 
% DATA_FILENAME = '../data/S020501C_CENTER_OUT_B.mat';
% TRAINING_PERIOD_START = 30;
% TRAINING_PERIOD_END   = 225;
% TEST_PERIOD_START     = 230;
% TEST_PERIOD_END       = 400;
% MINIMUM_SPIKES_PER_CELL = 800;
% FILTER_LENGTH         = 20;
% BIN_SIZE             = 0.05;
% SMOOTHING_WIDTH = 8;
% 
% % This is a function of the sample size of our digizing
% % tablet (1000 samples / cm) and the 3 low bits that get
% % removed as the samples are sent over a digital channel
% % into the neural recording hardware (a division by 8).
% 
% UNITS_PER_CM = 1000/8;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

addpath(FILTER_FUNCTION_PATH);

fprintf(1,'\nLoading neural and kinematic data...\n');
load(DATA_FILENAME);

% Retrieve the actual kinematic data from the experiment structure
x_goal = experiment.xpos(:,2);
x_t = experiment.xpos(:,1);
y_goal = experiment.ypos(:,2);
y_t = experiment.ypos(:,1);

if (min(x_goal) < 0) x_goal = x_goal - min(x_goal); end;
if (min(y_goal) < 0) y_goal = y_goal - min(y_goal); end;

% Convert to cm...
x_goal = x_goal / UNITS_PER_CM;
y_goal = y_goal / UNITS_PER_CM;

minx = min(x_goal);
maxx = max(x_goal);
miny = min(y_goal);
maxy = max(y_goal);
dist = sqrt( (maxx-minx)^2 + (maxy-miny)^2 );

% Now I need to bin the x and y coordinates so I have the same number of
% x's and y's, all lined up.  This does not have to be (and probably should not
% be) on the same bin granularity used by the filters.  I think I can just use the
% Matlab re-sampling functionality for this.
last_x_t = x_t(end);
first_x_t = x_t(1);
last_y_t = y_t(end);
first_y_t = y_t(1);

start_t = max(first_x_t,first_y_t);
end_t = min(last_x_t,last_y_t);
sample_resolution = 0.005;
t = [start_t:sample_resolution:end_t]';
x_resampled = interp1(x_t,x_goal,t);
y_resampled = interp1(y_t,y_goal,t);

% Generate predictions for cartesian x,y
experiment.xpos = [t x_resampled];
experiment.ypos = [t y_resampled];

fprintf(1,'\nBuilding filters on cartesian data...\n');
cartesian_f = make_linear_filter(experiment,[],BIN_SIZE,FILTER_LENGTH,TRAINING_PERIOD_START,TRAINING_PERIOD_END);

fprintf(1,'Finished building cartesian filter... generating predictions\n\n');
[cartesian_predict_x,cartesian_predict_y,cartesian_predict_t] = make_predictions(experiment,cartesian_f,TEST_PERIOD_START,TEST_PERIOD_END);

% Now convert to joint space
fprintf(1,'Doing inverse kinematics...');
[theta1, theta2] = inv_kin(x_base,y_base,x_resampled,y_resampled,a1,a2,elbow_up);

% Now build the joint-space filters...
%
% ...note that I'm over-writing the kinematic data here...
fprintf(1,'Building filters on joint-space data...\n');
experiment.xpos = [t theta1];
experiment.ypos = [t theta2];
angle_f = make_linear_filter(experiment,[],BIN_SIZE,FILTER_LENGTH,TRAINING_PERIOD_START,TRAINING_PERIOD_END);

% Now generate joint space predictions...
fprintf(1,'Finished building filters... generating predictions\n');
[angle_predict_theta1,angle_predict_theta2,angle_predict_t] = make_predictions(experiment,angle_f,TEST_PERIOD_START,TEST_PERIOD_END);

% Now reconstruct x and y from joint-space predictions...
fprintf(1,'Doing forward kinematics...\n\n');
[x_fwdkin,y_fwdkin,x_elbow,y_elbow] = forward_kin(x_base,y_base,angle_predict_theta1,angle_predict_theta2,a1,a2);

if (exist('smooth'))
  cartesian_predict_x = smooth(cartesian_predict_x,SMOOTHING_WIDTH);
  cartesian_predict_y = smooth(cartesian_predict_y,SMOOTHING_WIDTH);

  x_fwdkin = smooth(x_fwdkin,SMOOTHING_WIDTH);
  y_fwdkin = smooth(y_fwdkin,SMOOTHING_WIDTH);

  x_elbow = smooth(x_elbow,SMOOTHING_WIDTH);
  y_elbow = smooth(y_elbow,SMOOTHING_WIDTH);
else
  fprintf(1,'I really wanted to use the ''smooth'' function, but you don''t have it...\n');
end

% Generate static plots of both prediction sets
figure(1);
plot(angle_predict_t,x_fwdkin,'r',cartesian_predict_t,cartesian_predict_x,'g',x_t,x_goal,'b');
a = axis;
a(1) = min(angle_predict_t);
a(2) = max(angle_predict_t);
axis(a);
title('Predicted and actual x positions');
legend('Predicted x (joint-space)','Predicted x (cartesian space)','Actual x');
ylabel('x position (cm)');
xlabel('time (s)');

figure(2);
plot(angle_predict_t,y_fwdkin,'r',cartesian_predict_t,cartesian_predict_y,'g',y_t,y_goal,'b');
axis(a);
title('Predicted and actual y positions');
legend('Predicted y (joint-space)','Predicted y (cartesian space)','Actual y');
ylabel('y position (cm)');
xlabel('time (s)');

if (ANIMATE_RESULTS == 0)
  return;
end

% Render an animation of both predictions sets
base_pos = [x_base,y_base];
joint_predictions = [angle_predict_t x_elbow y_elbow x_fwdkin y_fwdkin];
cartesian_predictions = [cartesian_predict_t cartesian_predict_x cartesian_predict_y];
actual_positions = [t x_resampled y_resampled];
animate_invkin(base_pos,joint_predictions,0,cartesian_predictions,actual_positions);
