%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% function [filter] = make_kalman_filter(experiment,cell_list,bin_size,start_time,end_time,lag)
%
% Generates a kalman filter mapping neural firing rates to hand positions.  The algorithm
% used is derived from Wu et al, which should be packaged with this file.
%
% experiment is a struct containing neural and kinematic data, in the form 
% returned by plx_to_matlab.
%
% cell_list is a list of cells that should be used in building the filter,
% numbered as they appear in the plexon file.  There should be two columns:
% the first contains channels, the second contains units.
%
% if cells is empty ([]), all cells with a non-negligible number of spikes will be used.
%
% bin_size is the length of a bin, in seconds
%
% start_time (optional) is the time (in seconds) in the experiment that should
% be 'time 0' for the decoding filter
%
% end_time (optional) is the time (in seconds) at which the decoding filter
% should end.  The filter will actually end at the nearest bin-boundary before
% the specified end_time
%
% lag is the number of _bins_ that should separate "aligned" kinematic and spike records.
% To explain this, let's assume that k bins are required for building this filter (based on
% (start_time - end_time) / bin_size).  If the lag is zero, the filter sees bins 0 --> k-1 for
% both spikes and kinematics.  If the lag is 5, the filter sees bins 0 --> k-6 for spikes and
% bins 5 --> k-1 for kinematics.  This parameter defaults to 0.
%
% Note that this is a very simple implementation of lagging; it does not allow negative or
% non-uniform lags, although the latter has been shown to improve performance of the kalman
% filter.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Returns a filter structure, containing fields as defined in Wu et al:
%
% Q: the spike covariance matrix
% H: the optimal transformation from spikes to kinematics
% A: the state propagation matrix
% W: the kinematic noise matrix
% 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Dan Morris, Stanford University, 2003
% http://techhouse.brown.edu/~dmorris
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [filter] = make_kalman_filter(experiment,cell_list,bin_size,start_time,...
  end_time,lag,minimum_spikes_per_cell)

if (nargin < 3)
    fprintf(1,'\nUsage: make_kalman_filter experiment_struct cell_list bin_size [start_time] [end_time]\n\n');
    return;
end

if (nargin < 4)
    % start with the first spike by default
   start_time = experiment.spikes_by_time(1,3);
end

if (nargin < 5)
    % end with the last spike by default
    end_time = experiment.spikes_by_time(end,3);
end

if (nargin < 6)
    % default to no lag
    lag = 0;
end

if (nargin < 7)
  minimum_spikes_per_cell = 800;
end

% if no list of cells was provided, find 'good' cells
if (size(cell_list,1) == 0)
    cell_list = find_nonzero_cells(experiment,minimum_spikes_per_cell);
end

% Fill in some information so the filter will be more identifiable later
filter.cell_list = cell_list;
filter.bin_size = bin_size;
filter.start_time = start_time;
filter.end_time = end_time;
filter.filename = experiment.filename;
filter.lag = lag;

% compute the total number of bins in the specified time window
num_bins = floor( (end_time - start_time) / bin_size);
% num_bins = num_bins - lag

% This isn't really a new variable; I just use it in some places so the code will look
% more like the code from make_filter.m .  It's always the same as num_bins.
num_windows = num_bins;

num_cells = size(cell_list,1);

% I'm going to use the spike_times matrix in the experiment structure,
% which is 1-indexed.  The input cell_list is 0-indexed.
cell_list = cell_list + 1;

% the matrix unformatted_response is a matrix of spike counts, indexed as 
% (cell,bin)
[unformatted_response,R] = make_response_matrix(num_cells,num_bins,experiment,cell_list,bin_size,start_time,0);

% If there's a lag, cut off the last 'lag' bins
if (lag > 0)
    unformatted_response = unformatted_response(:,1:end-lag);
end

filter_length = 1;

% create the stimulus vectors (binned kinematic data) for x and y
fprintf(1,'Building stimulus vectors...\n');

s_x = zeros(num_windows,1);
s_y = s_x;

cur_x_index = 1;
cur_y_index = 1;

x_positions = experiment.xpos;
y_positions = experiment.ypos;

cur_x = -1;
cur_y = -1;
    
for(i=1:num_windows)
    window_start_time = start_time + (i-1)*bin_size;
    window_end_time = window_start_time + bin_size*filter_length;
    while(cur_x_index <= size(x_positions,1) & x_positions(cur_x_index,1) < window_end_time)
        cur_x = x_positions(cur_x_index,2);
        cur_x_index = cur_x_index + 1;
    end
    while(cur_y_index <= size(y_positions,1) & y_positions(cur_y_index,1) < window_end_time)
        cur_y = y_positions(cur_y_index,2);
        cur_y_index = cur_y_index + 1;
    end
    s_x(i) = cur_x;
    s_y(i) = cur_y;
end

% If there's a lag, cut off the first 'lag' bins
if (lag > 0)
    s_x = s_x(lag+1:end);
    s_y = s_y(lag+1:end);
end

% From here on, num_windows and num_bins will represent the number of bins actually stored
% in the filter, not the number of bins present in the whole time period
num_bins = num_bins - lag;
num_windows = num_bins;

% Store the stimulus and response matrices used to create this filter
filter.s_x = s_x;
filter.s_y = s_y;
filter.unformatted_response = unformatted_response;

% Now do the kalman magic

fprintf(1,'Preparing kalman filter coefficients...\n');

kinematics = [s_x s_y];

% Compute the mean x and y values, since the filter operates on normalized
% coordinates
center = mean(kinematics(:,1:2));
filter.center = center;

% Now normalize (subtract out the means)
kinematics(:,1:2) = kinematics(:,1:2)-ones(num_windows,1)*center;

% The experiment file provides only position; let's compute velocity
kinematics(:,3:4) = zeros(num_windows,2);
kinematics(2:num_windows,3:4) = diff(kinematics(:,1:2))/bin_size;

% The kinematics array now contains one row for each time step, with each
% row of the form :
% [x_pos y_pos x_vel y_vel]

% Compute the variables required for filtering, according to Wu et al
X2 = kinematics(2:num_windows,:)';
X1 = kinematics(1:num_windows-1,:)';

% Compute the necessary matrices for the 'system model' ;
% 
% x_{k+1} = A*x_k+w_k

% The least-squares-optimal transformation from x_i to x_(i+1)
A = X2*X1'*inv(X1*X1');

% The matrix of noise values for kinematics
W = ((X2 - A*X1)*((X2 - A*X1)')) / (num_windows - 1);

% A and W are both 4x4

% Compute the necessary matrices for the 'generative model' of neural firing :
%
% z_k = H*x_k+q_k

X = kinematics';
Z = unformatted_response;

fprintf(1,'Computing matrix inverse...\n');

% The least-squares-optimal transformation from x_i to z_i
% (the transformation from position to spikes)
H = Z*X'*(inv(X*X'));

% The covariance matrix for spikes
Q = ((Z - H*X)*((Z - H*X)')) / num_windows;

filter.Q = Q;
filter.H = H;
filter.A = A;
filter.W = W;
