%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function [filter] = 
%  make_linear_filter(experiment,cell_list,bin_size,filter_length,start_time,end_time)
%
% Generates a linear filter mapping neural firing rates to hand positions.  The
% algorithm used here is presented in Warland et al, which should be packaged with
% this file.
%
% experiment is a struct containing neural and kinematic data, in the form 
% returned by plx_to_matlab
%
% cell_list is a list of cells that should be used in building the filter,
% numbered as they appear in the plexon file.  There should be two columns:
% the first contains channels, the second contains units
%
% if cells is empty ([]), all cells with more than 0 spikes will be used.
%
% bin_size is the length of a bin, in seconds
%
% filter_length is the length of the decoding filter, in _bins_
%
% start_time (optional) is the time (in seconds) in the experiment that should
% be 'time 0' for the decoding filter
%
% end_time (optional) is the time (in seconds) at which the decoding filter
% should end.  The filter will actually end at the nearest bin-boundary before
% the specified end_time
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Returns a filter structure, containing fields:
%
% f_x and f_y : the coefficients for the x and y filters
% s_x and s_y : the stimulus vectors used for the x and y kinematics
% R           : the response matrix used to build the filter
% center      : the mean of the kinematic samples used to build the filter
% 
% s_x and s_y         : the stimulus vectors used for the x and y kinematics
% unformatted_reponse : the response matrix used to build the filter
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Dan Morris, Stanford University, 2003
% http://techhouse.brown.edu/~dmorris
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [filter] = make_linear_filter(experiment,cell_list,bin_size,filter_length,...
  start_time,end_time,minimum_spikes_per_cell)

if (nargin < 4)
    fprintf(1,'Invalid number of arguments...\n');
    help make_filter;
    return;
end

if (nargin < 7)
  minimum_spikes_per_cell = 800;
end

if (nargin < 6)
  % end with the last spike by default
  end_time = experiment.spikes_by_time(end,3);
end

if (nargin < 5)
    % start with the first spike by default
   start_time = experiment.spikes_by_time(1,3);
end

% If no cell list was specified, find 'good' cells in the experiment
if (size(cell_list,1) == 0)
    cell_list = find_nonzero_cells(experiment,minimum_spikes_per_cell);
end

% Fill in some information so the filter will be more identifiable later
filter.cell_list = cell_list;
filter.bin_size = bin_size;
filter.filter_length = filter_length;
filter.start_time = start_time;
filter.end_time = end_time;
filter.filename = experiment.filename;

% The total number of bins in the specified time window
num_bins = floor( (end_time - start_time) / bin_size);
num_cells = size(cell_list,1);

% I'm going to use the spike_times matrix in the experiment structure,
% which is 1-indexed.  The input cell_list is 0-indexed.
cell_list = cell_list + 1;

fprintf(1,'Building response matrix...\n');

% Get the response matrix, formatted according to Warland et al p2338
[unformatted_response,R] = make_response_matrix(num_cells,num_bins,experiment,cell_list,bin_size,start_time,filter_length);

% The number of complete filter windows in the response matrix
num_windows = num_bins - filter_length + 1;

% create the stimulus vectors (binned kinematic data) for x and y
fprintf(1,'Building stimulus vectors...\n');

s_x = zeros(num_windows,1);
s_y = s_x;

cur_x_index = 1;
cur_y_index = 1;

x_positions = experiment.xpos;
y_positions = experiment.ypos;

cur_x = -1;
cur_y = -1;
 
% Iterate through each bin, and find the x and y coordinates with the greatest time
% that's still in that bin.
for(i=1:num_windows)
    window_start_time = start_time + (i-1)*bin_size;
    window_end_time = window_start_time + bin_size*filter_length;
    while(cur_x_index <= size(x_positions,1) & x_positions(cur_x_index,1) < window_end_time)
        cur_x = x_positions(cur_x_index,2);
        cur_x_index = cur_x_index + 1;
    end
    while(cur_y_index <= size(y_positions,1) & y_positions(cur_y_index,1) < window_end_time)
        cur_y = y_positions(cur_y_index,2);
        cur_y_index = cur_y_index + 1;
    end
    s_x(i) = cur_x;
    s_y(i) = cur_y;
end

% Store the stimulus and response matrices in the filter struct
filter.s_x = s_x;
filter.s_y = s_y;
filter.R = R;

% At some point I was interested in the condition number of the response matrix that
% we're inverted, and how it varies with number of cells, window size, etc.
filter.cond = cond(R'*R);

% fprintf(1,'Computing inverse... condition number is %f\n',filter.cond);

fprintf(1,'Computing inverse...\n');

inverse_R = inv(R'*R);
filter.x_filter = inverse_R * (R'*s_x);
filter.y_filter = inverse_R * (R'*s_y);
