%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% function [unformatted_response,R] =
%   make_response_matrix(num_cells,num_bins,experiment,cell_list,bin_size,start_time,filter_length);
%
% This function is used to take data from an experiment struct (as returned by plx_to_matlab) and
% turn it into a nice cells x bins matrix _and_ a 'response matrix' in Warland et al form.
%
% This is used internally by the other functions that build filters and make predictions.
%
% Dan Morris, Stanford University, 2003
% http://techhouse.brown.edu/~dmorris
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [unformatted_response,R] = make_response_matrix(num_cells,num_bins,experiment,cell_list,bin_size,start_time,filter_length);

% This is not the main response matrix... this is the response for each cell
% in each bin, which will be reformatted into the response matrix.
%
% This matrix is indexed by (cell,bin)
unformatted_response = zeros(num_cells,num_bins);

fprintf(1,'Collecting responses for each cell...\n');

for(i=1:num_cells)
  
  % A large and unnecessary memory copy, for clarity
  %
  % If only the matlab interpreter could figure out that I'm not
  % going to modify this array, I would be able to modify it in-place
  % implicitly.  Oh well.
  %
  % Anyway these are the spike times for this cell
  spike_times = experiment.spike_times{cell_list(i,1),cell_list(i,2)};
    
  % My current position in this array
  curpos = 1;

  % Find the element corresponding to the first spike of interest
  while(curpos <= length(spike_times) & spike_times(curpos) < start_time)
    curpos = curpos + 1;
  end
  if (curpos > length(spike_times))
    fprintf(1,'Warning: no spikes in time range for cell %d (%d,%d)\n', ...
    i,cell_list(i,1),cell_list(i,2));
    continue;
  end
    
  % for each bin
  for(j=1:num_bins)        
    % count the number of spikes for this cell in this bin
    cur_bin_end = start_time + ((j-1) * bin_size);
    cur_bin_count = 0;
    while(curpos <= length(spike_times) & spike_times(curpos) < cur_bin_end)
      curpos = curpos + 1;
      cur_bin_count = cur_bin_count + 1;
    end
    unformatted_response(i,j) = cur_bin_count;
  end
end

if (filter_length == 0)
  R = [];
  return;
end

fprintf(1,'Building response matrix...\n');

% Now build the real response matrix, as formulated in Warland et al p2338
num_windows = num_bins - filter_length + 1;

% We initially ignore the constant column, to make indexing simpler
R = zeros(num_windows,filter_length*num_cells);

% build the matrix one row at a time
for(i=1:num_windows)
  % fill in this row one cell at a time
  for(j=1:num_cells)
    % the column indices where we're writing into the R matrix
    R_start_column = (j-1)*filter_length+1;
    R_end_column = R_start_column + filter_length - 1;

    % the columns in the spike count matrix
    spike_start_column = i;
    spike_end_column = i + filter_length - 1;

    R(i,R_start_column:R_end_column) = ...
      unformatted_response(j,spike_start_column:spike_end_column);
  end
end

% add the constant column
R = [ones(size(R,1),1) R];

