%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% function [experiment] = plx_to_matlab(plx_filename, mat_filename)
%
% This function converts a plexon file into a matlab structure... the
% point is to have the data from the .plx file available in a useful
% format and to avoid the _really_ slow process of reading in a .plx file
% every time you want to work with your data.
%
% It assumes the transformation of kinematic data into Plexon digital codes
% used in the Donoghue Lab at Brown.
%
% The function returns a single structure ('experiment'), which has
% following fields :
%
% header : information from the plexon file header that you probably aren't
% interested in.  The most exciting thing here is the date/time that the file
% was created.
%
% freq : the sampling rate during the experiment (hz)
%
% total_spikes : self-explanatory
% 
% spikes_by_time : all the spikes that were recorded during the experiment,
% sorted by time.  Three columns: channel, unit, time.  Channel and unit appear
% to be numbered starting from 1.  Note that the spikes in this array are
% _actually_ sorted by time; they're not in the order in which they were logged,
% which is out of temporal order in some cases.
% 
% spike_counts : an matrix of spike counts for each (channel,unit)
%
% spike_times : a cell array in which each {channel,unit} entry is a list
% of spike times for the corresponding cell.
%
% Note that for the previous two fields, I added one to each channel and unit
% when I read it in from the file, because of Matlab's silly 1-indexing.  Plexon
% is not very clear about whether the channels and units in the file are 0-
% or 1-indexed.
%
% events : all digital codes (not all events, just the digital codes)
% The first column is time, the second is the actual codes
% 
% xpos, ypos, neural_xpos, neural_ypos : Each matrix contains the corresponding
% kinematic/predicted kinematic information, extracted from the digital codes.
%
% For each matrix, the first column represented time and the second column
% represents x/y position.  These are also explicitly sorted by time.
%
% Dan Morris, Stanford University, 2003
% http://techhouse.brown.edu/~dmorris
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [experiment] = plx_to_matlab(plx_filename, mat_filename)

if(nargin < 1)
   disp('Usage: plx_to_matlab(plx_filename[,mat_filename])')
   return
end

write_matfile = 0;
if (nargin > 1)
    write_matfile = 1;
end

% Open the given plexon file, or ask the user which file he wants to use
if(length(plx_filename) == 0)
    [fname, pathname] = uigetfile('*.plx', 'Select a plx file');
    plx_filename = strcat(pathname, fname);
end

fid = fopen(plx_filename, 'r');

if(fid == -1)
    disp('cannot open file');
    return
end

disp(strcat('file=',plx_filename));
header = fread(fid, 64, 'int32');

% Fill in some useful meta-data in the experiment struct
experiment.header.version = header(2);
experiment.freq = header(35);  % frequency
experiment.header.ndsp = header(36);  % number of dsp channels
experiment.header.nevents = header(37); % number of external events
experiment.header.nslow = header(38);  % number of slow channels
experiment.header.npw = header(39);  % number of points in wave
experiment.header.npr = header(40);  % number of points before threshold
experiment.header.year = header(41);
experiment.header.month = header(42);
experiment.header.day = header(43);
experiment.header.hour = header(44);
experiment.header.minute = header(45);
experiment.header.second = header(46);
experiment.filename = plx_filename;

tscounts = fread(fid, [5, 130], 'int32');
wfcounts = fread(fid, [5, 130], 'int32');
evcounts = fread(fid, [1, 512], 'int32');

experiment.total_spikes = sum(sum(tscounts));
experiment.spikes_by_time = zeros(experiment.total_spikes,3);

% allocate all necessary memory ahead of time, to make things faster (Matlab is really
% bad at appending to arrays
experiment.spike_times = {};
experiment.spike_counts = [];

% Create empty lists for the spike records for each cell
for(i=1:130)
    for(j=1:5)
        experiment.spike_times{i,j} = zeros(1,tscounts(j,i));
        experiment.spike_counts(i,j) = 0;
    end
end

STROBE_CHANNEL = 257;

total_events = evcounts(STROBE_CHANNEL+1);
experiment.events = zeros(total_events,2);
event_num = 0;
current_spike = 0;

% skip variable headers
fseek(fid, 1020*experiment.header.ndsp + 296*experiment.header.nevents + 296*experiment.header.nslow,'cof');
record = 0;
dots = 0;

while feof(fid) == 0
  type = fread(fid, 1, 'int16');
  timestamp_upperbyte = fread(fid, 1, 'int16');

  if (timestamp_upperbyte > 0)
    disp(sprintf('Warning: nonzero timestamp upperbyte at channel %d, unit %d', ...
      channel,unit));
  end

  timestamp_lowerbytes = fread(fid, 1, 'int32');

  channel = fread(fid, 1, 'int16');
  unit = fread(fid, 1, 'int16');
  nwf = fread(fid, 1, 'int16');
  nwords = fread(fid, 1, 'int16');
  toread = nwords;
  if toread > 0
    wf = fread(fid, toread, 'int16');
  end

  % If this is a spike...
  if type == 1
   spike_num = experiment.spike_counts(channel+1,unit+1) + 1;
   experiment.spike_counts(channel+1,unit+1) = spike_num;
   experiment.spike_times{channel+1,unit+1}(spike_num) = ...
     timestamp_lowerbytes/experiment.freq;
   current_spike = current_spike + 1;
   experiment.spikes_by_time(current_spike,1) = channel;
   experiment.spikes_by_time(current_spike,2) = unit;
   experiment.spikes_by_time(current_spike,3) = ...
     timestamp_lowerbytes / experiment.freq;
  end

  % If this is a strobed code...
  if (type == 4 & channel == STROBE_CHANNEL)
    event_num = event_num+1;
    if (event_num > total_events) 
      disp(sprintf('Warning: event number %d, expectd only %d',event_num,total_events));
    end
    experiment.events(event_num,1) = timestamp_lowerbytes/experiment.freq;
    experiment.events(event_num,2) = unit;
  end

  record = record + 1;
  if (mod(record,10000) == 0)
    fprintf(1,sprintf('.',record));
    dots = dots + 1;
    if (dots == 60) 
      fprintf(1,'\n');
      dots = 0;
    end
  end
  if feof(fid) == 1
    break
  end
end

fclose(fid);

% sort the spikes_by_time array by time
[times,inds] = sort(experiment.spikes_by_time(:,3));
experiment.spikes_by_time = experiment.spikes_by_time(inds,:);


% Build the kinematic records from the digital codes
strobedcodes = experiment.events(:,2);
timestamps = experiment.events(:,1);

% The high-bit codes representing each data type
dio_mask = bitshift(1,12);
xpos_mask = bitshift(2,12);
ypos_mask = bitshift(3,12);

NEXT_XY_IS_NEURAL = 100;

dio_inds = find(strobedcodes >= dio_mask & strobedcodes < xpos_mask);
experiment.diocodes = [timestamps(dio_inds) (strobedcodes(dio_inds) - dio_mask)];

% Find all the codes that say 'the next pair of coordinates are neural predictions'
next_xy_is_neural_codes = find(strobedcodes==NEXT_XY_IS_NEURAL+dio_mask);

% Find the indices corresponding to neural predictions, so we can mask them out
% (becase we just want kinematics)
neural_xpos_inds = next_xy_is_neural_codes + 1;
neural_ypos_inds = next_xy_is_neural_codes + 2;
neuralmask = zeros(length(strobedcodes),1);
neuralmask(neural_xpos_inds) = 1;
neuralmask(neural_ypos_inds) = 1;

% Find the indices corresponding to hand positions
xpos_inds = find(strobedcodes >= xpos_mask & strobedcodes < ypos_mask & (~(neuralmask)));
ypos_inds = find(strobedcodes >= ypos_mask & (~(neuralmask)));

% Extract x and y positions and sort them by time
experiment.xpos = [timestamps(xpos_inds) (strobedcodes(xpos_inds) - xpos_mask)] ;
[times,inds] = sort(experiment.xpos(:,1));
experiment.xpos = experiment.xpos(inds,:);

experiment.ypos = [timestamps(ypos_inds) (strobedcodes(ypos_inds) - ypos_mask)] ;
[times,inds] = sort(experiment.ypos(:,1));
experiment.ypos = experiment.ypos(inds,:);

% Extract predicted x and y positions and sort them by time
experiment.neural_xpos = [timestamps(neural_xpos_inds) (strobedcodes(neural_xpos_inds) - xpos_mask)] ;
if (max(size(experiment.neural_xpos)) > 0)
  [times,inds] = sort(experiment.neural_xpos(:,1));
  experiment.neural_xpos = experiment.neural_xpos(inds,:);
end

experiment.neural_ypos = [timestamps(neural_ypos_inds) (strobedcodes(neural_ypos_inds) - ypos_mask)] ;
if (max(size(experiment.neural_ypos)) > 0)
  [times,inds] = sort(experiment.neural_ypos(:,1));
  experiment.neural_ypos = experiment.neural_ypos(inds,:);
end

% Write the result out to disk
if (write_matfile > 0)
  eval(sprintf('save %s experiment',mat_filename));
end
