%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% function [experiment] = tg2_to_matlab(tg2_spikes_filename, tg2_kin_filename, mat_filename)
%
% Example:
% tg2_to_matlab('data\spikes.tg2spikes','data\kinematics.tg2kin',0,'data\synthetic.mat');
%
% This function converts a tg2 kinematics file - containing kinematic data
% on the 'Neural source 2' channel - and a tg2 spikes file into a Matlab
% structure...
%
% The function returns a single structure ('experiment'), which has
% following fields (simplified from the format returned by plx_to_matlab):
%
% total_spikes : self-explanatory
% 
% spikes_by_time : all the spikes that were recorded during the experiment,
% sorted by time.  Three columns: channel, unit, time.  Channel and unit
% are numbered starting from 1.  Note that the spikes in this array are
% _actually_ sorted by time; they're not in the order in which they were logged,
% which is out of temporal order in some cases.
% 
% spike_counts : an matrix of spike counts for each (channel,unit)
%
% spike_times : a cell array in which each {channel,unit} entry is a list
% of spike times for the corresponding cell.
%
% xpos, ypos: kinematic information. 
%
% For xpos and ypos, the first column represents time and the second column
% represents x/y position.  These are also explicitly sorted by time.
%
% Dan Morris, Stanford University, 2003
% http://techhouse.brown.edu/~dmorris
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [experiment] = tg2_to_matlab(tg2_spikes_filename, tg2_kin_filename, target_number, mat_filename)

if(nargin < 3)
   disp('Usage: tg2_to_matlab(tg2_spikes_filename,tg2_kin_filename,target_number[,mat_filename])')
   return
end

write_matfile = 0;
if (nargin > 3)
    write_matfile = 1;
end

experiment.filename = tg2_spikes_filename;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Getting spike information...
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fprintf(1,'Reading spike data...\n');

% Extract the actual spikes from the data file...
spikes = dlmread(tg2_spikes_filename,'\t',3,0);
spike_rows = find(spikes(:,1)==0);
spikes = spikes(spike_rows,:);

experiment.total_spikes = size(spikes,1);

% Keep only [game time, channel, unit]

TG2SPIKES_TIME_COLUMN = 4;
TG2SPIKES_CHAN_COLUMN = 2;
TG2SPIKES_UNIT_COLUMN = 3;

spikes = spikes(:,[TG2SPIKES_TIME_COLUMN TG2SPIKES_CHAN_COLUMN TG2SPIKES_UNIT_COLUMN]);

% Throw out times < 0
goodspikes = find(spikes(:,1)>0);
spikes = spikes(goodspikes,:);

% Sort spikes by time
spikes = sortrows(spikes,1);

experiment.spikes_by_time = spikes;

% allocate all necessary memory ahead of time, to make things faster (Matlab is really
% bad at appending to arrays
experiment.spike_times = {};

MAX_CHANNEL = 100;
MAX_UNIT    = 5;
experiment.spike_counts = zeros(MAX_CHANNEL,MAX_UNIT);

fprintf(1,'Reformatting spike counts...\n');

% Fill in spike_counts and spike_times
for(i=1:MAX_CHANNEL)
    for(j=1:MAX_UNIT)
        myspike_indices = find(spikes(:,2)==i & spikes(:,3)==j);
        experiment.spike_counts(i,j) = length(myspike_indices);
        experiment.spike_times{i,j} = spikes(myspike_indices,1);
    end
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Getting kinematic information...
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fprintf(1,'Reading kinematic data...\n');

kin = dlmread(tg2_kin_filename,'\t',3,0);

% Device type == 'neural source 2'

TG2KIN_DEVICE_TYPE = -1;
TG2KIN_TARGET_NUM_COLUMN = 10;
TG2KIN_DEVICE_TYPE_COLUMN = 1;

fprintf(1,'Reformatting kinematic data...\n');

% Extract our kinematic data
kin_rows = find( ...
  (kin(:,TG2KIN_DEVICE_TYPE_COLUMN)==TG2KIN_DEVICE_TYPE) & ...
  (kin(:,TG2KIN_TARGET_NUM_COLUMN)==target_number) ...
);

TG2KIN_TIME_COLUMN = 8;
TG2KIN_X_COLUMN    = 2;
TG2KIN_Y_COLUMN    = 3;

% Extract [time,x,y]
kin = kin(kin_rows,[TG2KIN_TIME_COLUMN TG2KIN_X_COLUMN TG2KIN_Y_COLUMN]);

% Sort by time
kin = sortrows(kin,1);

experiment.xpos = kin(:,[1 2]);
experiment.ypos = kin(:,[1 3]);

% Write the result out to disk
if (write_matfile > 0)
  eval(sprintf('save %s experiment',mat_filename));
end