﻿using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

/****
 * 
 * This program takes a list of metadata about songs in our repertoire, including which instruments
 * play at the beginning and end of each song, which singers sing each song, etc, and a number of
 * other constraints (this song should appear at the beginning of a seat, this song should never
 * be in the third set, etc.), and generates a set list based on those constraints.  The approach
 * is randomized, so you can run it a few times to get something you like.
 * 
 * The program reads metadata from a .csv file; in practice I edit the metadata in an .xlsx file,
 * save to .csv, and do one-time manipulation (set up constraints specific to one show) in that 
 * .csv file.
 * 
 * The general approach is to use all the penalties and constraints set up in the metadata file
 * to greedily generate a reasonable set list (which will tend to have all sorts of bad song 
 * transitions toward the end of the set), then spend several million iterations randomly looking
 * for songs we can swap to improve that set list.  This is basically a simulated annealing approach;
 * early in our procedure we may make some known bad swaps, but at the end of the procedure we only
 * make swaps that improve the overall score.
 * 
 * The resulting set list is spit out to a text file and a nice templated html file.
 * 
 ****/

namespace SetListGenerator
{
    // Each vocalist will be assigned one of these values for each song
    // based on the metadata file.
    public enum VocalAssignment
    {
        VOCALASSIGNMENT_NONE,
        VOCALASSIGNMENT_PRIMARY,
        VOCALASSIGNMENT_SECONDARY,
        VOCALASSIGNMENT_UNKNOWN
    };

    // Each song will be assigned a bass tuning in which that song is played, or "any"
    public enum BassTuning
    {
        BASSTUNING_D,
        BASSTUNING_E,
        BASSTUNING_ANY
    };
    
    // Each song will be assigned a guitar on which that song is played, or "any"
    public enum GuitarAssignment
    {
        GUITARASSIGNMENT_STRAT,
        GUITARASSIGNMENT_HEAVY,
        GUITARASSIGNMENT_ANY,
        GUITARASSIGNMENT_UNKNOWN
    };

    // The data structure corresponding to a song
    public class Song
    {
        /*** All of these fields come right from the metadata file ***/
        public String title;
        public VocalAssignment paulVox;
        public VocalAssignment emVox;
        public VocalAssignment connorVox;
        public bool drumIntro;
        public bool keysCanWait;
        public bool noKeysAtEnd;
        public bool guitarCanWait;
        public bool noGuitarAtEnd;
        public GuitarAssignment gtr;
        public BassTuning bassTuning;
        public bool noBassAtEnd;
        public bool bassCanWait;
        public double setOpenerBonus;
        public double setCloserBonus;
        public double firstFiveSongsBonus;
        public double lastThreeSongsBonus;
        public double firstSetBonus;
        public double secondSetBonus;
        public double lastSetBonus;

        // These fields are populated as we generate our set list
        public double score;
        public bool guitarSelected = false;
    };
            
    class SetListGenerator
    {
        // Filled in from our spreadsheet
        public static int SET_BREAK_INDEX_1 = -1;
        public static int SET_BREAK_INDEX_2 = -1;
        public static int N_SONGS = -1;

        // Optionally filled in from our spreadsheet
        public static int TOTAL_ITERATIONS = 5000000;
        
        // File names
        public const String INPUT_FILE = @"song details.csv";
        public const String TMP_CSV_FILE = @"tmp.csv";
        public const String OUTPUT_TEXT_FILE = @"setlist.txt";
        public const String OUTPUT_HTML_FILE = @"setlist.html";
        public const String INPUT_HTML_TEMPLATE = @"template.html";
        public const String SETMARKER = "SETCONTENT";

        public static String[] SINGER_NAMES = new String[] { "Paul", "Emily", "Connor" };
        public static String[] GUITAR_NAMES = new String[] { "Strat", "Heavy", "Any" };

        // These penalties are hard-coded, but really should be included in the metadata file, since
        // one could imagine wanting to change them without recompiling.

        // How bad is it to "waste" a song that doesn't start with keyboards by using it at the
        // beginning of a set?  (These are typically valuable later to accelerate transitions.)
        static double PENALTY_KEYSCANWAIT_SETSTART = 10.0;

        // How bad is it to "waste" a song that doesn't start with guitars by using it at the
        // beginning of a set?  (These are typically valuable later to accelerate transitions.)
        static double PENALTY_GUITARCANWAIT_SETSTART = 10.0;

        // How bad is it to "waste" a song that has a drum-only intro by using it at the
        // beginning of a set?  (These are typically valuable later to accelerate transitions.)
        static double PENALTY_DRUMINTRO_SETSTART = 10.0;

        // How bad is it to use the same singer twice in a row?
        static double PENALTY_SINGER_REPEAT = 10.0;
        
        // How bad is it to wait for a guitar-change between songs?
        static double PENALTY_GUITARCHANGE = 10.0;

        // How bad is it to wait for a bass re-tune between songs?
        static double PENALTY_BASSRETUNE = 20.0;

        // How bad is it to wait for a keyboard patch change between songs?
        static double PENALTY_KEYSCONFLICT = 10.0;

        // How bad is it to have _any_ of the above conflicts (guitar, bass, keys) in the
        // first three songs of a set?  This penalty is applied in addition to the individual
        // penalties.
        static double PENALTY_ANYCONFLICT_IN_FIRST_THREE_OF_A_SET = 20.0;

        public const int SINGERINDEX_PAUL = 0;
        public const int SINGERINDEX_EMILY = 1;
        public const int SINGERINDEX_CONNOR = 2;

        // There is always some probability of making a song swap that improves
        // the score of the set; for good randomness, this starts at a non-unity
        // probability.
        public const double INITIAL_PROBABILITY_GOOD_SWAP = 0.95;
        public const double FINAL_PROBABILITY_GOOD_SWAP = 1.0;

        // There may be some small probability of making a song swap that makes 
        // the score of the set worse.  This should start small (if not zero) and
        // always end up at zero.
        public const double INITIAL_PROBABILITY_BAD_SWAP = 0.00;
        public const double FINAL_PROBABILITY_BAD_SWAP = 0.00;

        // When do the probabilities end up at their "final" state (in terms of progress
        // through our assigned number of iterations?
        public const double PROGRESS_AT_FINAL_PROBABILITIES = 0.9;

        // What is the probability of making a swap that has precisely zero impact on the
        // set score?
        public const double PROBABILITY_EQUAL_SWAP = 0.5;

        // This corresponds to the column ordering in the metadata file
        public enum SongDetailsColumns : int
        {
            COLUMN_TITLE = 0,
            COLUMN_PAULVOX,
            COLUMN_EMVOX,
            COLUMN_CONNORVOX,
            COLUMN_DRUMINTRO,
            COLUMN_KEYSCANWAIT,
            COLUMN_NOKEYSATEND,
            COLUMN_GUITARCANWAIT,
            COLUMN_NOGUITARATEND,
            COLUMN_GUITAR,
            COLUMN_BASSTUNING,
            COLUMN_NOBASSATEND,
            COLUMN_BASSCANWAIT,
            COLUMN_SETOPENERBONUS,
            COLUMN_SETCLOSERBONUS,
            COLUMN_FIRSTFIVESONGSBONUS,
            COLUMN_LASTTHREESONGSBONUS,
            COLUMN_FIRSTSETBONUS,
            COLUMN_SECONDSETBONUS,
            COLUMN_LASTSETBONUS,
            COLUMN_NUMCOLUMNS
        }

        public static VocalAssignment str2va(String s)
        {
            if (s == "x") return VocalAssignment.VOCALASSIGNMENT_PRIMARY;
            if (s == "1") return VocalAssignment.VOCALASSIGNMENT_PRIMARY;
            if (s == "2") return VocalAssignment.VOCALASSIGNMENT_SECONDARY;
            return VocalAssignment.VOCALASSIGNMENT_UNKNOWN;

        }

        public static BassTuning str2bt(String s)
        {
            if (s.Length == 0) return BassTuning.BASSTUNING_D;
            else if ((s.ToLower())[0] == 'e') return BassTuning.BASSTUNING_E;
            else return BassTuning.BASSTUNING_D;
        }

        public static GuitarAssignment str2ga(String s)
        {
            if (s == "1") return GuitarAssignment.GUITARASSIGNMENT_STRAT;
            if (s == "2") return GuitarAssignment.GUITARASSIGNMENT_HEAVY;
            if (s == "3") return GuitarAssignment.GUITARASSIGNMENT_ANY;
            return GuitarAssignment.GUITARASSIGNMENT_UNKNOWN;
        }

        public static String SingerName(Song s)
        {
            int index = SingerIndex(s);
            if (index == -1) return "none";
            return SINGER_NAMES[index];
        }

        public static String GuitarName(Song s)
        {
            return GUITAR_NAMES[(int)s.gtr];
        }

        public static int SingerIndex(Song s)
        {
            if (s == null) return -1;
            if (s.paulVox == VocalAssignment.VOCALASSIGNMENT_PRIMARY)
                return SINGERINDEX_PAUL;
            if (s.emVox == VocalAssignment.VOCALASSIGNMENT_PRIMARY) 
                return SINGERINDEX_EMILY;
            if (s.connorVox == VocalAssignment.VOCALASSIGNMENT_PRIMARY)
                return SINGERINDEX_CONNOR;
            return -1;
        }

        public static int CompareSongsByScore(Song s1, Song s2)
        {
            return s1.score.CompareTo(s2.score);
        }

        // Index refers to the _second_ argument
        static void ScoreSongPair(Song lastSong, Song s, int index)
        {
            s.score = 0.0;

            int indexInSet = -1;
            int whichSet = -1;

            if (SET_BREAK_INDEX_2 > 0 && index >= SET_BREAK_INDEX_2)
            {
                indexInSet = index - SET_BREAK_INDEX_2;
                whichSet = 2;
            }

            else if (SET_BREAK_INDEX_1 > 0 && index >= SET_BREAK_INDEX_1)
            {
                indexInSet = index - SET_BREAK_INDEX_1;
                whichSet = 1;
            }

            else
            {
                indexInSet = index;
                whichSet = 0;
            }

            
            // Penalize songs at the beginning of a set where the keys can wait
            if (lastSong == null)
            {
                if (s.keysCanWait) s.score -= PENALTY_KEYSCANWAIT_SETSTART;
            }

            // Penalize songs at the beginning of a set where the guitar can wait
            if (lastSong == null)
            {
                if (s.guitarCanWait) s.score -= PENALTY_GUITARCANWAIT_SETSTART;
            }

            // Penalize songs with a drum intro at the beginning of a set
            if (lastSong == null)
            {
                if (s.drumIntro) s.score -= PENALTY_DRUMINTRO_SETSTART;
            }

            // Penalize repeating singers, except at the beginning of a set
            // or when the previous song didn't really tap any of the singers
            int lastSingerIndex = SingerIndex(lastSong);
            int singerIndex = SingerIndex(s);

            if (lastSingerIndex == singerIndex &&
                singerIndex != -1)
            {
                s.score -= PENALTY_SINGER_REPEAT;
            }

            bool guitarConflict = false;
            bool keysConflict = false;
            bool bassConflict = false;

            // Penalize patch changes without extra time
            if (lastSong != null && s.keysCanWait == false && s.drumIntro == false
                && !(lastSong.noKeysAtEnd))
            {
                keysConflict = true;
                s.score -= PENALTY_KEYSCONFLICT;
            }

            // Penalize guitar changes, unless the guitar can wait or isn't around at the end
            // of the previous song
            GuitarAssignment lastGtr;
            if (lastSong == null) lastGtr = GuitarAssignment.GUITARASSIGNMENT_ANY;
            else lastGtr = lastSong.gtr;
            GuitarAssignment gtr = s.gtr;
            if (s.guitarCanWait == false &&
                lastGtr != gtr &&
                lastGtr != GuitarAssignment.GUITARASSIGNMENT_ANY &&
                gtr != GuitarAssignment.GUITARASSIGNMENT_ANY &&
                !(lastSong.noGuitarAtEnd))
            {
                guitarConflict = true;
                s.score -= PENALTY_GUITARCHANGE;
            }

            // Penalize bass changes, unless the bass can wait or isn't around at the end
            // of the previous song
            BassTuning lastBass;
            if (lastSong == null) lastBass = BassTuning.BASSTUNING_ANY;
            else lastBass = lastSong.bassTuning;
            BassTuning bassTuning = s.bassTuning;
            if (s.bassCanWait == false &&
                lastBass != bassTuning &&
                lastBass != BassTuning.BASSTUNING_ANY &&
                bassTuning != BassTuning.BASSTUNING_ANY &&
                !(lastSong.noBassAtEnd))
            {
                bassConflict = true;
                s.score -= PENALTY_BASSRETUNE;
            }

            // Have I committed this song to a guitar?
            if (s.gtr == GuitarAssignment.GUITARASSIGNMENT_ANY
                &&
                lastSong != null
                &&
                lastSong.gtr != GuitarAssignment.GUITARASSIGNMENT_ANY)
            {
                s.gtr = lastSong.gtr;
                s.guitarSelected = true;
            }

            // Penalize any conflicts early in a set
            if (indexInSet < 3 && (guitarConflict || bassConflict || keysConflict))
            {
                s.score -= PENALTY_ANYCONFLICT_IN_FIRST_THREE_OF_A_SET;
            }

            if (indexInSet == 0)
            {
                s.score += s.setOpenerBonus;
            }

            if (
                ((SET_BREAK_INDEX_1 > 0) && (index == SET_BREAK_INDEX_1 - 1))
                || 
                ((SET_BREAK_INDEX_2 > 0) && (index == SET_BREAK_INDEX_2 - 1))
                || 
                (index == N_SONGS - 1)
                )
            {
                s.score += s.setCloserBonus;
            }

            if (index > N_SONGS - 4)
            {
                s.score += s.lastThreeSongsBonus;
            }

            if (index <= 4)
            {
                s.score += s.firstFiveSongsBonus;
            }

            if (whichSet == 0) s.score += s.firstSetBonus;
            else if (whichSet == 1) s.score += s.secondSetBonus;
            else s.score += s.lastSetBonus;            
        }

        static Song ChooseNextSong(List<Song> songs, Song lastSong, int index)
        {
            foreach (Song s in songs)
            {
                ScoreSongPair(lastSong, s, index);
            }

            // Now sort by score (just for debugging)
            songs.Sort(new Comparison<Song>(CompareSongsByScore));

            // Find the max score
            double maxScore = double.NegativeInfinity;

            foreach (Song s in songs)
            {
                if (s.score > maxScore) maxScore = s.score;
            }

            List<Song> candidates = new List<Song>();
            
            foreach (Song s in songs)
            {
                if (s.score == maxScore) candidates.Add(s);
            }

            Console.WriteLine("Max score is {0}, shared by {1} songs",
                maxScore, candidates.Count);

            // Choose one
            Random r = new Random();
            int selectionIndex = r.Next(0, candidates.Count);
            Song selection = candidates[selectionIndex];
            return selection;
        }

        public static double EvaluateSongList(List<Song> songs)
        {
            Song lastSong = null;
            int index = 0;
            double score = 0.0;
            foreach (Song s in songs)
            {
                if (
                    ((SET_BREAK_INDEX_1 > 0) && (index == SET_BREAK_INDEX_1))
                    ||
                    ((SET_BREAK_INDEX_2 > 0) && (index == SET_BREAK_INDEX_2))
                    ||
                    (index == 0)
                    )
                {
                    lastSong = null;
                }

                ScoreSongPair(lastSong, s, index);

                // The score gets stored in the second song argument
                score += s.score;
                lastSong = s;
                index++;
            }

            return score;
        }

        // This is the main sorting algorithm, that randomly permutes the ordering
        // of songs, looking for overall score improvements.  The input list is modified
        // and returned.
        static List<Song> ChooseOptimalSongs(List<Song> songs)
        {            
            int totalIterations = TOTAL_ITERATIONS;
            Random r = new Random();
            int count = songs.Count;
            int lastPercent = 0;

            int goodSwaps = 0;
            int badSwaps = 0;
            int equalSwaps = 0;

            double p = 0.0;

            for (int iteration = 0; iteration < totalIterations; iteration++)
            {
                int percent = (int)Math.Floor(100.0 * ((double)(iteration) / (double)(totalIterations)));

                // Possibly print some stuff to the console
                if (percent != lastPercent)
                {
                    Console.WriteLine("{0} percent done", 100.0 * (double)iteration / (double)totalIterations);
                    Console.WriteLine("{0} good swaps ({1}), {2} bad swaps ({3}), {4} equal swaps",
                        goodSwaps, (double)goodSwaps / (double)iteration,
                        badSwaps, (double)badSwaps / (double)iteration,
                        equalSwaps);
                    double score = EvaluateSongList(songs);
                    Console.WriteLine("Current score: {0}\n", score);
                }
                lastPercent = percent;

                // Randomly choose two songs to swap
                int index1 = r.Next(0, count);
                int index2 = r.Next(0, count);

                // Don't swap a song with itself
                if (index1 == index2) continue;

                // What's the set list score if we don't swap these two songs?
                double scoreNoSwap = EvaluateSongList(songs);

                // Swap
                Song s = songs[index1];
                songs[index1] = songs[index2];
                songs[index2] = s;

                // What's the set list score if we do swap these two songs?
                double scoreWithSwap = EvaluateSongList(songs);

                // This is an equal-score swap
                if (scoreNoSwap == scoreWithSwap)
                {
                    p = r.NextDouble();
                    if (p < PROBABILITY_EQUAL_SWAP)
                    {
                        equalSwaps++;
                        continue;
                    }

                    // Swap back
                    s = songs[index1];
                    songs[index1] = songs[index2];
                    songs[index2] = s;
                    // Console.WriteLine("No impact...");
                    continue;
                }

                // Is this a good swap?
                bool goodSwap = (scoreWithSwap > scoreNoSwap);

                // Compute the current probabilities of making "good" and "bad" swaps
                // (which may change throughout our procedure)
                double probabilityGoodSwap = 1.0;
                double probabilityBadSwap = 0.0;
                double progress = (double)iteration / (double)totalIterations;
                if (progress > PROGRESS_AT_FINAL_PROBABILITIES)
                {
                    probabilityGoodSwap = FINAL_PROBABILITY_GOOD_SWAP;
                    probabilityBadSwap = FINAL_PROBABILITY_BAD_SWAP;
                }
                else
                {
                    probabilityGoodSwap =
                        progress * (FINAL_PROBABILITY_GOOD_SWAP - INITIAL_PROBABILITY_GOOD_SWAP)
                        + INITIAL_PROBABILITY_GOOD_SWAP;
                    probabilityBadSwap =
                        progress * (FINAL_PROBABILITY_BAD_SWAP - INITIAL_PROBABILITY_BAD_SWAP)
                        + INITIAL_PROBABILITY_BAD_SWAP;
                }

                p = r.NextDouble();
                bool swap = false;
                if (goodSwap)
                {
                    if (p < probabilityGoodSwap)
                    {
                        goodSwaps++;
                        //Console.WriteLine("Making a good swap ({0})...",
                           // (double)goodSwaps / (double)badSwaps);
                        swap = true;
                    }
                }
                else
                {
                    if (p < probabilityBadSwap)
                    {
                        badSwaps++;
                        //Console.WriteLine("Making a bad swap ({0})...",
                           // (double)goodSwaps / (double)badSwaps);
                        swap = true;
                    }
                }

                if (swap)
                {
                    // We're swapped right now
                }
                else
                {
                    // Swap back
                    s = songs[index1];
                    songs[index1] = songs[index2];
                    songs[index2] = s;
                }
            }

            return songs;
        }

        // This is a pure greedy search approach that we use to set up the initial list
        public static List<Song> GreedySearch(List<Song> songs)
        {
            Random r = new Random();

            List<Song> outSongs = new List<Song>();
            Song lastSong = null;
            int index = 0;
            while (songs.Count > 0)
            {
                if ((index == 0)
                    ||
                    ((SET_BREAK_INDEX_1 > 0) && (index == SET_BREAK_INDEX_1))
                    ||
                    ((SET_BREAK_INDEX_2 > 0) && (index == SET_BREAK_INDEX_2))
                    )
                {
                    lastSong = null;
                    if (index != 0)
                        Console.WriteLine("Set break...");
                }

                // Find a song that fits
                Song nextSong = ChooseNextSong(songs, lastSong, index);
                outSongs.Add(nextSong);
                songs.Remove(nextSong);
                lastSong = nextSong;

                index++;

                
            }

            return outSongs;
        }

        // Read all the song metadata from our .csv file
        static List<Song> ReadSongData()
        {
            List<Song> songs = new List<Song>();

            try
            {
                System.IO.File.Copy(INPUT_FILE, TMP_CSV_FILE, true);
                StreamReader sr = new StreamReader(TMP_CSV_FILE);
                int lineNumber = -1;
                String line = null;
                while (true)
                {
                    line = sr.ReadLine();
                    if (line == null) break;
                    lineNumber++;

                    // Dicard the header
                    if (lineNumber == 0) continue;

                    line = line.Trim();

                    // Discard blank lines
                    if (line.Length == 0) continue;

                    // Stop at the end token
                    if (line.StartsWith("END")) break;

                    char[] splitters = { ',' };
                    String[] tokens = line.Split(splitters);

                    if (line.StartsWith("SET_BREAK"))
                    {
                        List<String> nonEmptyTokens = new List<string>();
                        for (int col = 0; col < tokens.Length; col++)
                        {
                            if (tokens[col].Length == 0) continue;
                            nonEmptyTokens.Add(tokens[col]);
                        }
                        int whichBreak = int.Parse(nonEmptyTokens[1]);
                        int breakIndex = int.Parse(nonEmptyTokens[2]);

                        Console.WriteLine("Breaking set {0} at index {1}", whichBreak, breakIndex);

                        if (whichBreak == 0)
                        {
                            SET_BREAK_INDEX_1 = breakIndex;
                        }
                        else if (whichBreak == 1)
                        {
                            SET_BREAK_INDEX_2 = breakIndex;
                        }
                        continue;
                    }

                    if (tokens.Length != (int)SongDetailsColumns.COLUMN_NUMCOLUMNS)
                    {
                        Console.WriteLine("Illegal line: {0}", line);
                        return null;
                    }

                    Song s = new Song();
                    s.title = tokens[0].Trim();

                    // Ignore empty songs
                    if (s.title.Length == 0)
                    {
                        continue;
                    }

                    s.paulVox = str2va(tokens[(int)SongDetailsColumns.COLUMN_PAULVOX].Trim());
                    s.emVox = str2va(tokens[(int)SongDetailsColumns.COLUMN_EMVOX].Trim());
                    s.connorVox = str2va(tokens[(int)SongDetailsColumns.COLUMN_CONNORVOX].Trim());
                    s.drumIntro = (tokens[(int)SongDetailsColumns.COLUMN_DRUMINTRO].Trim().Length > 0);
                    s.keysCanWait = (tokens[(int)SongDetailsColumns.COLUMN_KEYSCANWAIT].Trim().Length > 0);
                    s.noKeysAtEnd = (tokens[(int)SongDetailsColumns.COLUMN_NOKEYSATEND].Trim().Length > 0);

                    s.guitarCanWait = (tokens[(int)SongDetailsColumns.COLUMN_GUITARCANWAIT].Trim().Length > 0);
                    s.noGuitarAtEnd = (tokens[(int)SongDetailsColumns.COLUMN_NOGUITARATEND].Trim().Length > 0);
                    s.gtr = str2ga(tokens[(int)SongDetailsColumns.COLUMN_GUITAR].Trim());

                    s.bassTuning = str2bt(tokens[(int)SongDetailsColumns.COLUMN_BASSTUNING].Trim());
                    s.noBassAtEnd = (tokens[(int)SongDetailsColumns.COLUMN_NOBASSATEND].Trim().Length > 0);
                    s.bassCanWait = (tokens[(int)SongDetailsColumns.COLUMN_BASSCANWAIT].Trim().Length > 0);

                    s.setOpenerBonus = (tokens[(int)SongDetailsColumns.COLUMN_SETOPENERBONUS].Trim().Length == 0)
                        ? 0.0 : Double.Parse(tokens[(int)SongDetailsColumns.COLUMN_SETOPENERBONUS]);
                    s.setCloserBonus = (tokens[(int)SongDetailsColumns.COLUMN_SETCLOSERBONUS].Trim().Length == 0)
                        ? 0.0 : Double.Parse(tokens[(int)SongDetailsColumns.COLUMN_SETCLOSERBONUS]);
                    s.firstFiveSongsBonus = (tokens[(int)SongDetailsColumns.COLUMN_FIRSTFIVESONGSBONUS].Trim().Length == 0)
                        ? 0.0 : Double.Parse(tokens[(int)SongDetailsColumns.COLUMN_FIRSTFIVESONGSBONUS]);
                    s.lastThreeSongsBonus = (tokens[(int)SongDetailsColumns.COLUMN_LASTTHREESONGSBONUS].Trim().Length == 0)
                        ? 0.0 : Double.Parse(tokens[(int)SongDetailsColumns.COLUMN_LASTTHREESONGSBONUS]);
                    s.firstSetBonus = (tokens[(int)SongDetailsColumns.COLUMN_FIRSTSETBONUS].Trim().Length == 0)
                        ? 0.0 : Double.Parse(tokens[(int)SongDetailsColumns.COLUMN_FIRSTSETBONUS]);
                    s.secondSetBonus = (tokens[(int)SongDetailsColumns.COLUMN_SECONDSETBONUS].Trim().Length == 0)
                        ? 0.0 : Double.Parse(tokens[(int)SongDetailsColumns.COLUMN_SECONDSETBONUS]);
                    s.lastSetBonus = (tokens[(int)SongDetailsColumns.COLUMN_LASTSETBONUS].Trim().Length == 0)
                        ? 0.0 : Double.Parse(tokens[(int)SongDetailsColumns.COLUMN_LASTSETBONUS]);

                    songs.Add(s);
                }
                N_SONGS = songs.Count;

                Console.WriteLine("Parsed {0} songs, breaking at {1} and {2}", N_SONGS, SET_BREAK_INDEX_1, SET_BREAK_INDEX_2);
                sr.Close();
            }
            catch (Exception e)
            {
                Console.WriteLine("Error processing file: {0}", e.ToString());
                return null;
            }

            Console.WriteLine("Parsed {0} songs", songs.Count);
            return songs;
        }

        // This is the top-level sorting routine (called from main()); it initializes
        // the set list with a greedy search, then does a randomized optimization.
        static List<Song> SortSongs(List<Song> songs)
        {
            List<Song> outSongs = GreedySearch(songs);
            outSongs = ChooseOptimalSongs(outSongs);
            return outSongs;
        }

        // Write the set list out to a pretty html file
        static void WriteSetList(List<Song> songs)
        {
            Song curSong = null;
                
            try
            {
                StreamWriter sw = new StreamWriter(OUTPUT_TEXT_FILE);
                
                String templateString = File.ReadAllText(INPUT_HTML_TEMPLATE);

                String currentSetString = "";
                String matchString = "";

                int index = 0;
                Song lastSong = null;
                double score = EvaluateSongList(songs);
                sw.WriteLine("Total score: {0}\n", score);

                foreach (Song s in songs)
                {
                    curSong = s;
                    if (index == 0 
                        || 
                        ((SET_BREAK_INDEX_1 > 0) && (index == SET_BREAK_INDEX_1))
                        || 
                        ((SET_BREAK_INDEX_2 > 0) && (index == SET_BREAK_INDEX_2))
                        )
                    {                        
                        Console.WriteLine("");
                        sw.WriteLine("");
                        lastSong = null;
                    }

                    if (
                        ((SET_BREAK_INDEX_1 > 0) && (index == SET_BREAK_INDEX_1))
                        || 
                        ((SET_BREAK_INDEX_2 > 0) && (index == SET_BREAK_INDEX_2))
                        )
                    {
                        matchString = SETMARKER + ((index == SET_BREAK_INDEX_1) ? "1" : "2");
                        templateString = templateString.Replace(matchString, currentSetString);
                        currentSetString = "";
                    }

                    String currentSongString = "<p class=\"song\">";
                    String singer = SingerName(s);                    
                    String guitar = GuitarName(s);
                    
                    currentSongString += s.title;
                    if (SingerIndex(s) !=  SINGERINDEX_EMILY 
                        && 
                        SingerIndex(s) != -1)
                        currentSongString += " (" + singer + ")";

                    Console.WriteLine("Writing song {0}", s.title);
                    sw.Write("{0,-30}", s.title);
                    sw.Write("{0,-10}", singer);
                    sw.Write("{0,-10}", guitar);
                    sw.Write("{0,-5}", s.score);
                    if (s.guitarSelected) sw.Write(" (guitar selected)");
                    
                    if (lastSong != null && s.keysCanWait == false && s.drumIntro == false && lastSong.noKeysAtEnd == false) 
                    {
                        sw.Write(" (keys conflict)");
                        currentSongString += " (kc)";
                    }
                    
                    if (lastSong != null && s.guitarCanWait == false && s.drumIntro == false
                        && s.gtr != lastSong.gtr && 
                        s.gtr != GuitarAssignment.GUITARASSIGNMENT_ANY &&
                        lastSong.gtr != GuitarAssignment.GUITARASSIGNMENT_ANY
                        && lastSong.noGuitarAtEnd == false
                        ) 
                    {
                        sw.Write(" (guitar conflict)");                    
                        currentSongString += " (gc)";
                    }

                    // if (s.bassTuning != BassTuning.BASSTUNING_D)
                       // currentSongString += " (bass in D)";

                    if (lastSong != null && s.bassCanWait == false && s.drumIntro == false
                        && s.bassTuning != lastSong.bassTuning &&
                        s.bassTuning != BassTuning.BASSTUNING_ANY &&
                        lastSong.bassTuning != BassTuning.BASSTUNING_ANY
                        && lastSong.noBassAtEnd == false
                        ) 
                    {
                        sw.Write(" (bass retune)");
                        currentSongString += " (bc)";
                    }

                    lastSong = s;
                    sw.WriteLine("");
                    currentSongString += "</p>";
                    currentSongString += "\n";
                    currentSetString += currentSongString;
                    index++;
                }
                sw.Close();

                matchString = SETMARKER + "3";
                templateString = templateString.Replace(matchString, currentSetString);
                currentSetString = "";

                File.WriteAllText(OUTPUT_HTML_FILE, templateString);
            }
            catch (Exception e)
            {
                Console.WriteLine("Error at song {0} writing file: {1}", curSong == null ? "null" : curSong.title, 
                    e.ToString());
                return;
            }

            Console.WriteLine("Done writing set list...");
        }

        static void Main(string[] args)
        {
            
            List<Song> songs = ReadSongData();
            if (songs == null) return;

            List<Song> outSongs = SortSongs(songs);
            WriteSetList(outSongs);
        }
    }
}
